from dataclasses import dataclass
from crm.agency_cabinet.documents.server.src.db import (
    Payment, Contract, Invoice, db
)

from crm.agency_cabinet.documents.common import structs


@dataclass
class ListPayments:
    async def __call__(self, params: structs.ListPaymentsInput) -> [structs.Payment]:
        query = db.select(
            [
                Payment.id,
                Payment.eid,
                Payment.amount,
                Payment.currency,
                Payment.date,
                Invoice.eid.label('invoice_eid'),
                Invoice.id.label('invoice_id'),
                Invoice.contract_id
            ]
        ).select_from(
            Payment.join(Invoice).join(Contract)
        ).where(
            Contract.agency_id == params.agency_id
        )

        if params.contract_id:
            query = query.where(
                Invoice.contract_id == params.contract_id
            )

        if params.invoice_id:
            query = query.where(
                Invoice.id == params.invoice_id
            )

        if params.date_from:
            query = query.where(
                Payment.date >= params.date_from
            )

        if params.date_to:
            query = query.where(
                Payment.date < params.date_to
            )

        if params.limit:
            query = query.limit(params.limit)

        if params.offset:
            query = query.offset(params.offset)

        payments = await query.order_by(Payment.date.desc()).gino.all()

        return [
            structs.Payment(
                payment_id=payment.id,
                eid=payment.eid,
                invoice_id=payment.invoice_id,
                amount=payment.amount,
                currency=payment.currency,
                date=payment.date,
                invoice_eid=payment.invoice_eid,
                contract_id=payment.contract_id,
            ) for payment in payments
        ]
