import typing
from library.python.monitoring.solo.objects.solomon.v2 import Graph, Element, Cluster
from library.python.monitoring.solo.objects.solomon.sensor import Sensor
from crm.agency_cabinet.common.monitoring.project import agency_cabinet_solomon_project
from crm.agency_cabinet.common.monitoring.clusters import testing, prestable, prod
from crm.agency_cabinet.gateway.monitoring.registry.sensors.metrics import (
    prod_gateway_response_rate_ok,
    prod_gateway_response_rate_client_error,
    prod_gateway_response_rate_error,
    prod_gateway_response_time,
    testing_gateway_response_rate_ok,
    testing_gateway_response_rate_client_error,
    testing_gateway_response_rate_error,
    testing_gateway_response_time,
    prestable_gateway_response_rate_ok,
    prestable_gateway_response_rate_client_error,
    prestable_gateway_response_rate_error,
    prestable_gateway_response_time
)


def build_api_rps_graph(cluster: Cluster):
    metrics_map: typing.Mapping[str, Sensor] = {
        f'{testing.name}_rate_ok': testing_gateway_response_rate_ok,
        f'{testing.name}_rate_error': testing_gateway_response_rate_error,
        f'{testing.name}_rate_client_error': testing_gateway_response_rate_client_error,

        f'{prestable.name}_rate_ok': prestable_gateway_response_rate_ok,
        f'{prestable.name}_rate_error': prestable_gateway_response_rate_error,
        f'{prestable.name}_rate_client_error': prestable_gateway_response_rate_client_error,

        f'{prod.name}_rate_ok': prod_gateway_response_rate_ok,
        f'{prod.name}_rate_error': prod_gateway_response_rate_error,
        f'{prod.name}_rate_client_error': prod_gateway_response_rate_client_error,
    }
    return Graph(
        id=f"{cluster.name}_gateway_rps_api_graph",
        project_id=agency_cabinet_solomon_project.id,
        name=f"Gateway API RPS Graph ({cluster.name})",
        min="0",
        transform="NONE",
        hide_no_data=False,
        normalize=False,
        ignore_inf=False,
        max_points=0,
        downsampling="BY_POINTS",
        scale="NATURAL",
        over_lines_transform="NONE",
        elements={
            Element(
                title="Success {{host}}",
                type="EXPRESSION",
                expression="""
                    sum({{sensor="{sensor_name}", path="/api/*", cluster="{cluster_name}"}}) by host
                """.format(sensor_name=metrics_map[f'{cluster.name}_rate_ok'].name, cluster_name=cluster.name),
                stack="rps",
                area=True
            ),
            Element(
                title="Client error {{host}}",
                type="EXPRESSION",
                expression="""
                    sum({{sensor="{sensor_name}", path="/api/*", cluster="{cluster_name}"}}) by host
                """.format(sensor_name=metrics_map[f'{cluster.name}_rate_client_error'].name, cluster_name=cluster.name),
                stack="rps",
                area=True
            ),
            Element(
                title="Server error {{host}}",
                type="EXPRESSION",
                expression="""
                    sum({{sensor="{sensor_name}", path="/api/*", cluster="{cluster_name}"}}) by host
                """.format(sensor_name=metrics_map[f'{cluster.name}_rate_error'].name, cluster_name=cluster.name),
                stack="rps",
                area=True
            ),
        }
    )


def build_response_time_hist_graph(cluster: Cluster):
    metrics_map: typing.Mapping[str, Sensor] = {
        f'{prestable.name}': prestable_gateway_response_time,
        f'{testing.name}': testing_gateway_response_time,
        f'{prod.name}': prod_gateway_response_time,
    }
    return Graph(
        id=f"{cluster.name}_gateway_response_time_hist_graph",
        project_id=agency_cabinet_solomon_project.id,
        name=f"Gateway Response Time Hist ({cluster.name})",
        min="0",
        color_scheme="GRADIENT",
        transform="NONE",
        hide_no_data=False,
        normalize=False,
        ignore_inf=False,
        max_points=0,
        downsampling="BY_POINTS",
        scale="NATURAL",
        over_lines_transform="NONE",
        elements={
            Element(
                title="Average API Response Time Distribution (bin {{bin}} ms)",
                type="EXPRESSION",
                expression="""
                        series_avg(
                        ['sensor', 'bin'],
                        {{sensor="{sensor_name}", service="gateway", path="/api/*", cluster="{cluster_name}", bin="*"}}
                    )
                    """.format(sensor_name=metrics_map[cluster.name].name, cluster_name=cluster.name),
                stack="response_distribution",
                area=True
            ),
        }
    )


testing_gateway_rps_api_graph = build_api_rps_graph(testing)
prod_gateway_rps_api_graph = build_api_rps_graph(prod)
prestable_gateway_rps_api_graph = build_api_rps_graph(prestable)

testing_time_distribution_graph = build_response_time_hist_graph(testing)
prestable_time_distribution_graph = build_response_time_hist_graph(prestable)
prod_time_distribution_graph = build_response_time_hist_graph(prod)


register = [
    testing_gateway_rps_api_graph,
    prod_gateway_rps_api_graph,
    prestable_gateway_rps_api_graph,
    testing_time_distribution_graph,
    prod_time_distribution_graph,
    prestable_time_distribution_graph
]
