import contextlib
import logging.config
import functools
import typing
from aioboto3 import Session
from aiobotocore.config import AioConfig
from aiohttp import web
from crm.agency_cabinet.common.monitoring.utils.registry import setup_monitoring
from crm.agency_cabinet.common.monitoring.utils.server import setup_metrics_registry
from crm.agency_cabinet.common.aiohttp_logging import RequestIdContextAccessLogger
from crm.agency_cabinet.common.server.common.config import MdsConfig
from crm.agency_cabinet.common.server.common.logging_config import get_logging_config
from crm.agency_cabinet.common.server.common.tvm import TvmClient, get_tvm_client
from crm.agency_cabinet.common.server.web.apispec import setup_aiohttp_apispec, setup_aiohttp_subapp_apispec
from crm.agency_cabinet.common.server.web.middlewares import (
    RequestIdMiddleware,
    TvmServiceMiddleware,
    TvmUserMiddleware,
    RateMonitoringMiddleware
)
from crm.agency_cabinet.gateway.server.src.middlewares import ValidationMiddleware, ErrorMiddleware
from crm.agency_cabinet.gateway.server.src.middlewares.error import process_validation_error
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery


from .config import GatewayConfig
from .handlers import (
    OrdReportsCollection,
    OrdCampaignsCollection,
    AgenciesCollection,
    BonusesCollection,
    CalculatorCollection,
    CertificatesCollection,
    InfraCollection,
    ReportsCollection,
    RewardsCollection,
    DocumentsCollection,
    RoleModelCollection,
    ContractsCollection,
    AnalyticsCollection,
    InvoicesCollection,
    PaymentsCollection,
    ActsCollection,
    AgreementsCollection,
    OrdClientsCollection,
    OrdActsCollection,
    OrdOrganizationsCollection,
    OrdContractsCollection,
    OrdInvitesCollection,
    OrdClientRowsCollection,
)


def init_boto3_session(mds_cfg: MdsConfig) -> typing.Optional[Session]:
    if mds_cfg.access_key_id and mds_cfg.secret_access_key:
        return Session(aws_access_key_id=mds_cfg.access_key_id, aws_secret_access_key=mds_cfg.secret_access_key)


async def setup_s3_client(app: web.Application, endpoint_url, boto3_session, mds_bucket) -> None:
    if endpoint_url is None or boto3_session is None:
        return

    context_stack = contextlib.AsyncExitStack()
    app['context_stack'] = context_stack

    app['mds_bucket'] = mds_bucket

    app['s3_resource'] = await context_stack.enter_async_context(
        boto3_session.resource('s3', endpoint_url=endpoint_url,
                               config=AioConfig(s3={'addressing_style': 'virtual'}))
    )
    app['s3_client'] = await context_stack.enter_async_context(
        boto3_session.client('s3', endpoint_url=endpoint_url,
                             config=AioConfig(s3={'addressing_style': 'virtual'}))
    )


async def shutdown_s3_client(app: web.Application) -> None:
    if 'context_stack' in app:
        await app['context_stack'].aclose()


def run_server():
    cfg = GatewayConfig.from_environ()
    logging.config.dictConfig(get_logging_config(cfg.verbose))

    async def _setup(c):
        sd = ServiceDiscovery(c.amqp_url)
        await sd.connect()
        return await setup_app(c, sd, get_tvm_client(c.tvm2))

    web.run_app(_setup(cfg), port=cfg.port, access_log_class=RequestIdContextAccessLogger)


async def setup_app(cfg: GatewayConfig, service_discovery: ServiceDiscovery, tvm_client: TvmClient) -> web.Application:
    app = web.Application(
        middlewares=[
            RateMonitoringMiddleware(),
            RequestIdMiddleware(),
            ErrorMiddleware(),
        ]
    )

    setup_metrics_registry_bounded = functools.partial(setup_metrics_registry, metric_registry=setup_monitoring())
    app.on_startup.append(setup_metrics_registry_bounded)

    boto3_session = init_boto3_session(cfg.mds_cfg)
    setup_s3_client_url_bounded = functools.partial(setup_s3_client,
                                                    endpoint_url=cfg.mds_cfg.endpoint_url, boto3_session=boto3_session,
                                                    mds_bucket=cfg.mds_cfg.bucket)
    app.on_startup.append(setup_s3_client_url_bounded)
    app.on_shutdown.append(shutdown_s3_client)

    infra_handlers = InfraCollection(service_discovery)
    app.add_routes(
        [
            web.get('/ping', infra_handlers.ping, allow_head=False),
            web.get('/health_check', infra_handlers.health_check, allow_head=False),
            web.get('/metrics', infra_handlers.metrics, allow_head=False)  # TODO: hide this api method?????
        ]
    )

    api = await setup_api_app(cfg, service_discovery, tvm_client, [setup_metrics_registry_bounded])
    app.add_subapp('/api/', api)
    setup_aiohttp_apispec(
        app=app,
        title="Agency Cabinet API",
        version="v1",
        url="/docs/swagger.json",  # дока по всем роутам app и его subapp
        swagger_path="/docs",
        error_callback=process_validation_error,
    )
    setup_aiohttp_subapp_apispec(
        app,
        api
    )

    return app


async def setup_api_app(
    cfg: GatewayConfig,
    service_discovery: ServiceDiscovery,
    tvm_client: TvmClient,
    on_startup: list[typing.Callable]
) -> web.Application:

    api = web.Application(
        middlewares=[
            RateMonitoringMiddleware(),
            RequestIdMiddleware(),
            ErrorMiddleware(),
            ValidationMiddleware(),
            TvmServiceMiddleware(tvm_client=tvm_client, allowed_clients=cfg.allowed_clients,
                                 development_mode=cfg.development_mode),
            TvmUserMiddleware(tvm_client=tvm_client, development_mode=cfg.development_mode),
        ]
    )
    for hook in on_startup:
        api.on_startup.append(hook)

    agencies_handlers = AgenciesCollection(service_discovery)
    calculator_handlers = CalculatorCollection(service_discovery)
    rewards_handlers = RewardsCollection(service_discovery)
    bonuses_handler = BonusesCollection(service_discovery)
    reports_handler = ReportsCollection(service_discovery)
    certificates_handler = CertificatesCollection(service_discovery)
    documents_handlers = DocumentsCollection(service_discovery)
    role_handlers = RoleModelCollection(service_discovery)
    contracts_handlers = ContractsCollection(service_discovery)
    analytics_handlers = AnalyticsCollection(service_discovery)
    invoices_handlers = InvoicesCollection(service_discovery)
    payments_handlers = PaymentsCollection(service_discovery)
    acts_handlers = ActsCollection(service_discovery)
    agreements_handlers = AgreementsCollection(service_discovery)
    ord_reports_handlers = OrdReportsCollection(service_discovery)
    ord_campaigns_handlers = OrdCampaignsCollection(service_discovery)
    ord_clients_handlers = OrdClientsCollection(service_discovery)
    ord_acts_handlers = OrdActsCollection(service_discovery)
    ord_organizations_handlers = OrdOrganizationsCollection(service_discovery)
    ord_contracts_handlers = OrdContractsCollection(service_discovery)
    ord_invites_handlers = OrdInvitesCollection(service_discovery)
    ord_client_rows_handlers = OrdClientRowsCollection(service_discovery)

    api.add_routes(
        [
            # OLD AGENCIES ROUTES
            web.get('/agencies', agencies_handlers.list_agencies, allow_head=False),
            web.get('/partners', role_handlers.list_available_partners, allow_head=False),
            web.get(r'/partners/{agency_id:\d+}/agency', agencies_handlers.agency_info, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}', agencies_handlers.agency_info, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/rewards', rewards_handlers.list_rewards, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/rewards/{reward_id:\d+}',
                    rewards_handlers.reward_info, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/rewards/dashboard', rewards_handlers.dashboard, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/contracts', agencies_handlers.list_contracts, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/clients', agencies_handlers.list_clients, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/calculator/meta', calculator_handlers.get_meta, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/calculator/init', calculator_handlers.get_init_data, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/calculator', calculator_handlers.list_available_calculators, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/reports', reports_handler.list_reports, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/reports/{report_id:\d+}',
                    reports_handler.report_info, allow_head=False),
            web.delete(r'/agencies/{agency_id:\d+}/reports/{report_id:\d+}', reports_handler.delete_report),
            web.post(r'/agencies/{agency_id:\d+}/reports/{report_id:\d+}/delete',
                     reports_handler.delete_report),  # temp
            web.get(r'/agencies/{agency_id:\d+}/reports/{report_id:\d+}/download_url', reports_handler.report_url,
                    allow_head=False),
            web.post(r'/agencies/{agency_id:\d+}/reports', reports_handler.create_report),
            web.get(r'/agencies/{agency_id:\d+}/documents/{document_id:\d+}/download_url',
                    documents_handlers.document_url, allow_head=False),

            web.get(r'/agencies/{agency_id:\d+}/bonuses', bonuses_handler.list_bonuses, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/bonuses/settings',
                    bonuses_handler.get_bonuses_settings, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/bonuses/{client_id:\d+}',
                    bonuses_handler.fetch_client_bonuses, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/bonuses/{client_id:\d+}/graph',
                    bonuses_handler.fetch_client_bonuses_graph, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/bonuses/{client_id:\d+}/history_log',
                    bonuses_handler.fetch_client_bonuses_details, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/bonuses/reports', bonuses_handler.list_bonuses_reports,
                    allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/bonuses/reports/{report_id:\d+}/download_url',
                    bonuses_handler.report_url, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/bonuses/reports/{report_id:\d+}',
                    bonuses_handler.report_info, allow_head=False),
            web.post(r'/agencies/{agency_id:\d+}/bonuses/reports', bonuses_handler.create_report),
            web.post(r'/agencies/{agency_id:\d+}/bonuses/reports/{report_id:\d+}/delete',
                     bonuses_handler.delete_report),
            web.delete(r'/agencies/{agency_id:\d+}/bonuses/reports/{report_id:\d+}',
                       bonuses_handler.delete_report),  # temp
            web.get(r'/agencies/{agency_id:\d+}/bonuses/cashback_programs',
                    bonuses_handler.list_cashback_programs, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/certificates',
                    certificates_handler.list_agency_certificates, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/certificates/history',
                    certificates_handler.fetch_agency_certificates_history, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/employees/certificates',
                    certificates_handler.list_employees_certificates, allow_head=False),
            web.get(
                r'/agencies/{agency_id:\d+}/certificates/direct/details',
                certificates_handler.get_agency_certificate_details,
                allow_head=False,
            ),
            # roles
            web.get(r'/agencies/{agency_id:\d+}/users', role_handlers.list_users, allow_head=False),
            web.post(r'/agencies/{agency_id:\d+}/users/{user_id:\d+}', role_handlers.edit_user),
            web.get(r'/agencies/{agency_id:\d+}/users/suggested', role_handlers.list_suggest_users, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/users/roles', role_handlers.list_roles, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/roles', role_handlers.list_user_roles, allow_head=False),

            # analytics
            web.get(
                r'/agencies/{agency_id:\d+}/analytics/graphs/market_situation',
                analytics_handlers.get_market_situation_graph,
                allow_head=False
            ),
            web.get(
                r'/agencies/{agency_id:\d+}/analytics/graphs/grades_distribution',
                analytics_handlers.get_grades_distribution_graph,
                allow_head=False
            ),
            web.get(
                r'/agencies/{agency_id:\d+}/analytics/graphs/active_clients',
                analytics_handlers.get_active_clients_graph,
                allow_head=False
            ),
            web.get(
                r'/agencies/{agency_id:\d+}/analytics/graphs/clients_increase',
                analytics_handlers.get_clients_increase_graph,
                allow_head=False
            ),


            # docs
            web.get(r'/agencies/{agency_id:\d+}/documents/contracts', contracts_handlers.list_contracts,
                    allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/documents/contracts/{contract_id:\d+}',
                    contracts_handlers.get_contract_info, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/documents/contracts/{contract_id:\d+}/download_url',
                    contracts_handlers.get_contract_url, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/documents/agreements', agreements_handlers.list_agreements,
                    allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/documents/agreements/{agreement_id:\d+}/download_url',
                    agreements_handlers.get_agreement_url, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/documents/invoices', invoices_handlers.list_invoices,
                    allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/documents/invoices/{invoice_id:\d+}', invoices_handlers.get_invoice_info,
                    allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/documents/invoices/{invoice_id:\d+}/download_url',
                    invoices_handlers.get_invoice_url, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/documents/invoices/factures/{facture_id:\d+}/download_url',
                    invoices_handlers.get_facture_url, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/documents/payments', payments_handlers.list_payments, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/documents/acts', acts_handlers.list_acts, allow_head=False),
            web.get(r'/agencies/{agency_id:\d+}/documents/acts/{act_id:\d+}/download_url',
                    acts_handlers.get_act_url, allow_head=False),

            # ord
            web.get(
                r'/agencies/{agency_id:\d+}/ord/reports',
                ord_reports_handlers.list_reports,
                allow_head=False
            ),
            web.post(
                r'/agencies/{agency_id:\d+}/ord/reports',
                ord_reports_handlers.create_report,
            ),
            web.get(
                r'/agencies/{agency_id:\d+}/ord/reports/{report_id:\d+}',
                ord_reports_handlers.report_info,
                allow_head=False
            ),
            web.post(
                r'/agencies/{agency_id:\d+}/ord/reports/{report_id:\d+}/delete',
                ord_reports_handlers.delete_report,
            ),
            web.get(
                r'/agencies/{agency_id:\d+}/ord/reports/{report_id:\d+}/clients',
                ord_clients_handlers.list_clients,
                allow_head=False
            ),
            web.post(
                r'/agencies/{agency_id:\d+}/ord/reports/{report_id:\d+}/clients',
                ord_clients_handlers.create_client,
            ),
            web.get(
                r'/agencies/{agency_id:\d+}/ord/reports/{report_id:\d+}/clients/{client_id:\d+}/rows',
                ord_client_rows_handlers.list_client_rows,
                allow_head=False
            ),
            web.post(
                r'/agencies/{agency_id:\d+}/ord/reports/{report_id}/clients/{client_id}/rows/{row_id}',
                ord_client_rows_handlers.edit_client_row,
            ),
            web.post(
                r'/agencies/{agency_id}/ord/reports/{report_id}/export',
                ord_reports_handlers.report_export,
            ),
            web.get(
                r'/agencies/{agency_id}/ord/reports/{report_id}/export/{report_export_id}',
                ord_reports_handlers.get_report_export_info,
                allow_head=False
            ),
            web.get(
                r'/agencies/{agency_id}/ord/reports/{report_id}/export/{report_export_id}/download_url',
                ord_reports_handlers.get_report_url,
                allow_head=False
            ),
            web.post(
                r'/agencies/{agency_id:\d+}/ord/reports/{report_id:\d+}/send',
                ord_reports_handlers.send_report
            ),
            web.post(
                r'/agencies/{agency_id:\d+}/ord/reports/{report_id:\d+}/import',
                ord_reports_handlers.import_data
            ),
            web.get(
                r'/agencies/{agency_id:\d+}/ord/reports/{report_id:\d+}/lock',
                ord_reports_handlers.get_lock_status,
                allow_head=False
            ),
            web.get(
                r'/agencies/{agency_id:\d+}/ord/reports/{report_id:\d+}/clients/{client_id:\d+}/campaigns',
                ord_campaigns_handlers.list_campaigns,
                allow_head=False
            ),
            web.get(
                r'/agencies/{agency_id:\d+}/ord/reports/{report_id:\d+}/acts',
                ord_acts_handlers.get_acts
            ),
            web.get(
                r'/agencies/{agency_id:\d+}/ord/reports/{report_id:\d+}/clients/{client_id:\d+}',
                ord_clients_handlers.get_client_short_info,
                allow_head=False
            ),
            web.post(
                r'/agencies/{agency_id:\d+}/ord/reports/{report_id:\d+}/acts',
                ord_acts_handlers.add_act
            ),
            web.post(
                r'/agencies/{agency_id:\d+}/ord/reports/{report_id:\d+}/acts/{act_id:\d+}',
                ord_acts_handlers.edit_act
            ),
            web.get(
                r'/agencies/{agency_id:\d+}/ord/organizations',
                ord_organizations_handlers.list_organizations,
                allow_head=False
            ),
            web.get(
                r'/agencies/{agency_id:\d+}/ord/contracts',
                ord_contracts_handlers.list_contracts,
                allow_head=False
            ),
            web.get(
                r'/agencies/{agency_id:\d+}/ord/invites',
                ord_invites_handlers.get_invites
            )
        ]
    )

    return api
