from typing import Union

from crm.agency_cabinet.gateway.server.src.structs import ErrorDetails, ErrorMessage

__all__ = [
    'AccessDenied',
    'NotFound',
    'ProcedureException',
    'Unprocessable',
    'InternalServerError',
    'BadRequest',
    'Conflict'
]


class ProcedureException(Exception):  # TODO: move to common

    default_http_code = None
    default_error_code = None
    default_message = None

    def __init__(self, *messages: Union[str, ErrorMessage], http_code: int = None, error_code: str = None):
        if http_code is None:
            http_code = self.default_http_code
        if error_code is None:
            error_code = self.default_error_code
        if not messages and self.default_message:
            messages = [self.default_message]
        elif not messages:
            self.message = 'Missing argument \'messages\''
            self.http_code = http_code
            super().__init__(self.message)
            return

        # to allow to modify, because initial type is 'tuple', which is immutable
        messages = list(messages)

        for i in range(len(messages)):
            message = messages[i]

            if not isinstance(message, str) and not isinstance(message, ErrorMessage):
                self.message = 'Expected \'str\' or \'ErrorMessage\', found: {}'.format(type(message))
                self.http_code = http_code
                super().__init__(self.message)
                return
            elif isinstance(message, str):
                messages[i] = ErrorMessage(text=message)

        self.message = 'HTTP code: {http_code} - {error_code} -> {messages}' \
            .format(http_code=http_code,
                    error_code=error_code,
                    messages='; '.join([message.process_message() for message in messages]))

        self.error_details = ErrorDetails(http_code=http_code, error_code=error_code, messages=messages)
        self.http_code = http_code
        super().__init__(self.message)


class AccessDenied(ProcedureException):
    default_http_code = 403
    default_error_code = 'ACCESS_DENIED'
    default_message = 'You don\'t have access'


class NotFound(ProcedureException):
    default_http_code = 404
    default_error_code = 'NOT_FOUND'
    default_message = 'Object is not found'


class Unprocessable(ProcedureException):
    default_http_code = 422
    default_error_code = 'VALIDATION_ERROR'
    default_message = 'Can\'t process request due to bad parameters'


class InternalServerError(ProcedureException):
    default_http_code = 500
    default_error_code = 'INTERNAL_SERVER_ERROR'
    default_message = 'Unknown'


class BadRequest(ProcedureException):
    default_http_code = 400
    default_error_code = 'BAD_REQUEST'
    default_message = 'Can\'t parse request'


class Conflict(ProcedureException):
    default_http_code = 409
    default_error_code = 'CONFLICT'
    default_message = 'Request conflict'


MAP_CODE_TO_EXCEPTION = {
    AccessDenied.default_http_code: AccessDenied,
    NotFound.default_http_code: NotFound,
    Unprocessable.default_http_code: Unprocessable,
    InternalServerError.default_http_code: InternalServerError,
    BadRequest.default_http_code: BadRequest,
    Conflict.default_http_code: Conflict
}
