from marshmallow import EXCLUDE
from aiohttp import web
from aiohttp_apispec import docs, querystring_schema

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.handlers.common import development_headers_docs, json_response
from crm.agency_cabinet.gateway.server.src.handlers.schemas import agencies_schemas
from crm.agency_cabinet.gateway.server.src.procedures import agencies_procedures


class AgenciesCollection:
    def __init__(self, sd: ServiceDiscovery, use_partner_id=False):
        self._list_agencies = agencies_procedures.ListAgencies(sd)
        self._agency_info = agencies_procedures.AgencyInfo(sd)
        self._list_contracts = agencies_procedures.ListContracts(sd)
        self._list_clients = agencies_procedures.ListClients(sd)

        self.use_partner_id = use_partner_id

    @docs(
        tags=['agencies'],
        summary='Список агентств, доступных пользователю',
        responses={
            200: {
                'schema': agencies_schemas.AgenciesListSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @development_headers_docs
    async def list_agencies(self, request: web.Request) -> web.Response:
        agencies = await self._list_agencies(request['yandex_uid'])

        data = {
            'agencies': agencies
        }
        return json_response(data=agencies_schemas.AgenciesListSchema().dump(data))

    @docs(
        tags=['agencies'],
        summary='Информация об агентстве',
        responses={
            200: {
                'schema': agencies_schemas.AgencyInfoSchema,
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            },
            404: {
                'description': 'Not found',
            }
        }
    )
    @development_headers_docs
    async def agency_info(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        agency_info = await self._agency_info(request['yandex_uid'], agency_id)
        agency_info.contracts = await self._list_contracts(request['yandex_uid'], agency_id)
        return json_response(data=agencies_schemas.AgencyInfoSchema().dump(agency_info))

    @docs(
        tags=['rewards_contracts'],
        summary='Список договоров',
        responses={
            200: {
                'schema': agencies_schemas.ContractsListSchema,
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @development_headers_docs
    async def list_contracts(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        contracts = await self._list_contracts(request['yandex_uid'], agency_id)

        data = {
            'contracts': contracts
        }
        return json_response(data=agencies_schemas.ContractsListSchema().dump(data))

    @docs(
        tags=['clients'],
        summary='Список клиентов',
        responses={
            200: {
                'schema': agencies_schemas.ClientsListSchema,
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @querystring_schema(agencies_schemas.GetClientsInfoSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def list_clients(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])  # noqa
        clients = await self._list_clients(request['yandex_uid'], agency_id, **request['querystring'])

        data = {
            'clients': clients
        }
        return json_response(data=agencies_schemas.ClientsListSchema().dump(data))
