from aiohttp import web
from aiohttp_apispec import docs, querystring_schema
from marshmallow import EXCLUDE
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.handlers.common import development_headers_docs, json_response
from crm.agency_cabinet.gateway.server.src.handlers.schemas import analytics_schemas
from crm.agency_cabinet.gateway.server.src.procedures import analytics_procedures


class AnalyticsCollection:
    def __init__(self, sd: ServiceDiscovery):
        self. _get_market_situation_data = analytics_procedures.GetMarketSituationData(sd)
        self._get_grades_distribution_data = analytics_procedures.GetGradesDistributionData(sd)
        self._get_active_clients_data = analytics_procedures.GetGetActiveClientsData(sd)
        self._get_clients_increase_data = analytics_procedures.GetGetClientsIncreaseData(sd)

    @docs(
        tags=['analytics'],
        summary='График ситуации на рынке (средние чеки)',
        responses={
            200: {
                'schema': analytics_schemas.MarketSituationGraphSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @querystring_schema(analytics_schemas.RequestAnalyticsSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def get_market_situation_graph(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        data = await self._get_market_situation_data(request['yandex_uid'], agency_id, **request['querystring'])
        # TODO: move to schema/pre_dum if need
        return json_response(data=analytics_schemas.MarketSituationGraphSchema().dump(
            {
                'other': data.other,
                'current_at_left_date': data.current_at_left_date,
                'current_at_right_date': data.current_at_right_date,
                'description': {
                    'agencies_with_less_average_budget_percent': data.percent_less
                }
            }
        ))

    @docs(
        tags=['analytics'],
        summary='Круговая диаграмма распределения грейдов',
        responses={
            200: {
                'schema': analytics_schemas.ClientByGradesGraphSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @querystring_schema(analytics_schemas.RequestAnalyticsSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def get_grades_distribution_graph(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        data = await self._get_grades_distribution_data(request['yandex_uid'], agency_id, **request['querystring'])

        return json_response(data=analytics_schemas.ClientByGradesGraphSchema().dump(
            {
                'current': data.current,
                'other': data.other,
                'description': {
                    'average_budget': data.median_budget_current,
                    'is_higher_than_other_agencies': data.median_budget_current >= data.median_budget_other,
                    # TODO: move to schema logic
                    'most_customers_grade': max(data.current, key=lambda e: e.percent).grade if data.current else '0-50'
                }
            }
        ))

    @docs(
        tags=['analytics'],
        summary='График активных клиентов',
        responses={
            200: {
                'schema': analytics_schemas.ActiveClientsGraphSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @querystring_schema(analytics_schemas.RequestAnalyticsSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def get_active_clients_graph(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        data = await self._get_active_clients_data(request['yandex_uid'], agency_id, **request['querystring'])
        # TODO: move to schema/pre_dum if need
        return json_response(data=analytics_schemas.ActiveClientsGraphSchema().dump(
            {
                'other': data.other,
                'current': data.current,
                'description': {
                    'percentage_of_agencies_with_less_clients': data.percent_less
                }
            }
        ))

    @docs(
        tags=['analytics'],
        summary='График прироста клиентов',
        responses={
            200: {
                'schema': analytics_schemas.IncreaseGraphSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @querystring_schema(analytics_schemas.RequestAnalyticsSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def get_clients_increase_graph(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        data = await self._get_clients_increase_data(request['yandex_uid'], agency_id, **request['querystring'])
        # TODO: move to schema/pre_dum if need
        return json_response(data=analytics_schemas.IncreaseGraphSchema().dump({
            'other': data.other,
            'current_at_left_date': data.current_at_left_date,
            'current_at_right_date': data.current_at_right_date,
            'description': {
                'percentage_of_agencies_with_less_increase': data.percent_less,
                'increase_for_period': analytics_schemas.IncreaseDatasetSchema().compute_increase_percent(
                    data.current_at_right_date
                )

            }
        }))
