import datetime

from aiohttp import web
from aiohttp_apispec import docs, querystring_schema, json_schema
from crm.agency_cabinet.client_bonuses.common.structs import BonusType, ClientType
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.exceptions import InternalServerError
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound
from crm.agency_cabinet.gateway.server.src.handlers.common import (
    development_headers_docs,
    json_response,
)
from crm.agency_cabinet.gateway.server.src.handlers.schemas import bonuses_schemas, common_schemas
from crm.agency_cabinet.gateway.server.src.procedures import bonuses_procedures
from marshmallow import EXCLUDE


class BonusesCollection:
    def __init__(self, sd: ServiceDiscovery):
        self._list_bonuses = bonuses_procedures.ListBonuses(sd)
        self._get_bonuses_settings = bonuses_procedures.GetBonusesSettings(sd)
        self._fetch_client_bonuses_details = (
            bonuses_procedures.FetchClientBonusesDetails(sd)
        )
        self._fetch_client_bonuses_graph = bonuses_procedures.FetchClientBonusesGraph(sd)
        self._list_bonuses_reports = bonuses_procedures.ListBonusesReports(sd)
        self._get_report_url = bonuses_procedures.GetClientBonusesReportUrl(sd)
        self._get_report_info = bonuses_procedures.GetClientBonusesReportInfo(sd)
        self._create_report = bonuses_procedures.CreateReport(sd)
        self._delete_report = bonuses_procedures.DeleteReport(sd)

        self._list_cashback_programs = bonuses_procedures.ListCashbackPrograms(sd)

    @docs(
        tags=["bonuses"],
        summary="Список информации о клиентских бонусах",
        responses={
            200: [bonuses_schemas.ClientBonusSchema(many=True)],
            422: {"name": "Missing data for required field."},
        },
        parameters=[
            {
                "in": "path",
                "name": "agency_id",
                "schema": {"type": "int"},
                "required": "true",
            }
        ],
    )
    @querystring_schema(bonuses_schemas.ListClientBonusesParamsSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def list_bonuses(self, request: web.Request) -> web.Response:
        agency_id = request.match_info["agency_id"]
        params = request['querystring']

        clients_bonuses = await self._list_bonuses(
            yandex_uid=request["yandex_uid"],
            agency_id=int(agency_id),
            limit=params["limit"],
            offset=params["offset"],
            client_type=params["client_type"],
            bonus_type=params["bonus_type"],
            datetime_start=params["datetime_start"],
            datetime_end=params["datetime_end"],
            search_query=params.get("search_query"),
        )

        return json_response(
            data=bonuses_schemas.ClientBonusSchema(many=True).dump(clients_bonuses)
        )

    @docs(
        tags=["bonuses"],
        summary="Настройки клиентских бонусов",
        responses={
            200: {bonuses_schemas.ClientBonusSettingsSchema},
            422: {"name": "Missing data for required field."},
        },
        parameters=[
            {
                "in": "path",
                "name": "agency_id",
                "schema": {"type": "int"},
                "required": "true",
            }
        ],
    )
    @development_headers_docs
    async def get_bonuses_settings(self, request: web.Request) -> web.Response:
        agency_id = request.match_info["agency_id"]

        clients_bonuses_settings = await self._get_bonuses_settings(
            yandex_uid=request["yandex_uid"],
            agency_id=int(agency_id)
        )
        return json_response(
            data=bonuses_schemas.ClientBonusSettingsSchema().dump(clients_bonuses_settings)
        )

    @docs(
        tags=["bonuses"],
        summary="Информация о бонусах конкретного клиента",
        responses={
            200: {"schema": bonuses_schemas.ClientBonusSchema()},
            422: {"name": "Missing data for required field."},
            404: {"text": "Can't find bonuses for client"}
        },
        parameters=[
            {
                "in": "path",
                "name": "agency_id",
                "schema": {"type": "int"},
                "required": "true",
            },
            {
                "in": "path",
                "name": "client_id",
                "schema": {"type": "int"},
                "required": "true",
            }
        ],
    )
    @development_headers_docs
    async def fetch_client_bonuses(self, request: web.Request) -> web.Response:
        agency_id = request.match_info["agency_id"]
        client_id = int(request.match_info["client_id"])
        # TODO: use separate method in future?
        clients_bonuses = await self._list_bonuses(
            yandex_uid=request["yandex_uid"],
            agency_id=int(agency_id),
            limit=0,
            offset=0,
            client_type=ClientType.ALL,
            bonus_type=BonusType.ALL,
            datetime_start=datetime.datetime.min.replace(tzinfo=datetime.timezone.utc),
            datetime_end=datetime.datetime.now(tz=datetime.timezone.utc),
            search_query=str(client_id),
        )
        client_bonus = next((bonus for bonus in clients_bonuses if bonus.client_id == client_id), None)
        if client_bonus:
            return json_response(
                data=bonuses_schemas.ClientBonusSchema().dump(client_bonus)
            )
        raise NotFound(f'Can\'t find bonuses for client {client_id}')

    @docs(
        tags=["bonuses"],
        summary="Детальная информация о бонусах клиента агентства",
        responses={
            200: [bonuses_schemas.ClientBonusDetailSchema(many=True)],
            400: {"name": "Missing data for required field."},
            404: {"text": "Object is not found"},
        },
        parameters=[
            {
                "in": "path",
                "name": "agency_id",
                "schema": {"type": "int"},
                "required": "true",
            },
            {
                "in": "path",
                "name": "client_id",
                "schema": {"type": "int"},
                "required": "true",
            },
        ],
    )
    @querystring_schema(bonuses_schemas.FetchClientBonusesDetailsParamsSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def fetch_client_bonuses_details(self, request: web.Request) -> web.Response:
        agency_id = request.match_info["agency_id"]
        client_id = request.match_info["client_id"]
        params = request['querystring']

        details = await self._fetch_client_bonuses_details(
            yandex_uid=request["yandex_uid"],
            agency_id=int(agency_id),
            client_id=int(client_id),
            datetime_start=params["datetime_start"],
            datetime_end=params["datetime_end"],
        )

        return json_response(
            data=bonuses_schemas.ClientBonusDetailSchema(many=True).dump(details)
        )

    @docs(
        tags=["bonuses"],
        summary="Информация для построения графика для клиента в бонусах клиентского агентства",
        responses={
            200: [bonuses_schemas.ClientBonusesGraphSchema()],
            400: {"name": "Missing data for required field."},
            404: {"text": "Object is not found"},
        },
        parameters=[
            {
                "in": "path",
                "name": "agency_id",
                "schema": {"type": "int"},
                "required": "true",
            },
            {
                "in": "path",
                "name": "client_id",
                "schema": {"type": "int"},
                "required": "true",
            },
        ],
    )
    @development_headers_docs
    async def fetch_client_bonuses_graph(self, request: web.Request) -> web.Response:
        agency_id = request.match_info["agency_id"]
        client_id = request.match_info["client_id"]

        details = await self._fetch_client_bonuses_graph(
            yandex_uid=request["yandex_uid"],
            agency_id=int(agency_id),
            client_id=int(client_id),
        )

        return json_response(
            data=bonuses_schemas.ClientBonusesGraphSchema().dump(details)
        )

    @docs(
        tags=['bonuses', 'reports'],
        summary='Список отчетов по бонусам',
        responses={
            200: {
                'schema': bonuses_schemas.ClientBonusesReportInfoSchema(many=True),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def list_bonuses_reports(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        reports = await self._list_bonuses_reports(
            yandex_uid=request['yandex_uid'],
            agency_id=agency_id
        )

        return json_response(
            data=bonuses_schemas.ClientBonusesReportInfoSchema(many=True).dump(reports)
        )

    @docs(
        tags=['bonuses', 'reports'],
        summary='Ссылка для скачивания отчета',
        responses={
            200: {
                'schema': bonuses_schemas.ClientBonusesReportUrlSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            404: {
                'description': 'Not found',
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def report_url(self, request: web.Request) -> web.Response:
        report_id = int(request.match_info['report_id'])
        agency_id = int(request.match_info['agency_id'])
        url = await self._get_report_url(request['yandex_uid'], agency_id, report_id)
        return json_response(
            data=bonuses_schemas.ClientBonusesReportUrlSchema().dump({'url': url}),
            headers={'Cache-Control': 'no-cache'}
        )

    @docs(
        tags=['bonuses', 'reports'],
        summary='Информация об отчёте',
        responses={
            200: {
                'schema': bonuses_schemas.ClientBonusesReportInfoSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            404: {
                'description': 'Not found',
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def report_info(self, request: web.Request) -> web.Response:
        report_id = int(request.match_info['report_id'])
        agency_id = int(request.match_info['agency_id'])
        report_info = await self._get_report_info(request['yandex_uid'], agency_id, report_id)
        return json_response(
            data=bonuses_schemas.ClientBonusesReportInfoSchema().dump(report_info),
            headers={'Cache-Control': 'no-cache'}
        )

    @docs(
        tags=['bonuses', 'reports'],
        summary='Создание отчета по бонусам',
        responses={
            200: {
                'schema': bonuses_schemas.ClientBonusesReportInfoSchema(),
                'description': 'Ok'
            },
            400: {
                'description': 'Bad request',
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @json_schema(bonuses_schemas.CreateBonusesReportInfoSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def create_report(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        report_json = request['json']
        result = await self._create_report(
            yandex_uid=request['yandex_uid'],
            agency_id=agency_id,
            report_json=report_json)
        return json_response(data=bonuses_schemas.ClientBonusesReportInfoSchema().dump(result))

    @docs(
        tags=['reports', 'bonuses'],
        summary='Удаление отчета по бонусам',
        responses={
            200: {
                'description': 'Ok',
                'schema': common_schemas.DeleteStatusSchema()
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            404: {
                'description': 'Not found',
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @json_schema(common_schemas.EmptySchema())
    @development_headers_docs
    async def delete_report(self, request: web.Request) -> web.Response:
        report_id = int(request.match_info['report_id'])
        agency_id = int(request.match_info['agency_id'])
        result = await self._delete_report(
            yandex_uid=request['yandex_uid'],
            agency_id=agency_id,
            report_id=report_id)
        if result:
            return json_response(data=common_schemas.DeleteStatusSchema().dump({}))
        raise InternalServerError('Unsuccessful report delete operation')

    @docs(
        tags=['bonuses'],
        summary='Список актуальных программ',
        responses={
            200: {
                'schema': bonuses_schemas.CashbackProgramsListSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            404: {
                'description': 'Not found',
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def list_cashback_programs(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        cashback_programs = await self._list_cashback_programs(request['yandex_uid'], agency_id)
        return json_response(
            data=bonuses_schemas.CashbackProgramsListSchema().dump(
                {
                    'programs': cashback_programs,
                }
            ),
        )
