from aiohttp import web
from aiohttp_apispec import docs, querystring_schema
from marshmallow import EXCLUDE

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.handlers.common import (
    development_headers_docs,
    json_response,
)
from crm.agency_cabinet.gateway.server.src.handlers.schemas import certificates_schemas
from crm.agency_cabinet.gateway.server.src.procedures import certificates_procedures


class CertificatesCollection:
    def __init__(self, sd: ServiceDiscovery):
        self._list_employees_certificates = (
            certificates_procedures.ListEmployeesCertificates(sd)
        )
        self._list_agency_certificates = certificates_procedures.ListAgencyCertificates(
            sd
        )
        self._fetch_agency_certificates_history = (
            certificates_procedures.FetchAgencyCertificatesHistory(sd)
        )
        self._get_agency_certificate_details = (
            certificates_procedures.GetAgencyCertificatesDetails(sd)
        )

    @docs(
        tags=["certificates"],
        summary="Список сертификатов сотрудников агентства",
        responses={
            200: [certificates_schemas.AgencyCertificateSchema(many=True)],
            403: {
                "description": "Forbidden",
            },
        },
        parameters=[
            {
                "in": "path",
                "name": "agency_id",
                "schema": {"type": "int"},
                "required": "true",
            }
        ],
    )
    @querystring_schema(certificates_schemas.EmployeeCertificatesParamsSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def list_employees_certificates(self, request: web.Request) -> web.Response:
        agency_id = request.match_info["agency_id"]

        params = request['querystring']

        certificates_info = await self._list_employees_certificates(
            yandex_uid=request["yandex_uid"],
            agency_id=int(agency_id),
            offset=params["offset"],
            limit=params["limit"],
            project=params.get("project"),
            search_query=params.get("search_query"),
            status=params.get("status"),
        )

        return json_response(
            data=certificates_schemas.CertifiedEmployeeSchema(many=True).dump(
                certificates_info
            )
        )

    @docs(
        tags=["certificates"],
        summary="Список сертификатов агентства",
        responses={
            200: [certificates_schemas.AgencyCertificateSchema(many=True)],
            422: {"name": "Missing data for required field."},
            403: {"description": "Forbidden"},
            404: {"description": "Not found"},
        },
        parameters=[
            {
                "in": "path",
                "name": "agency_id",
                "schema": {"type": "int"},
                "required": "true",
            },
        ],
    )
    @development_headers_docs
    async def list_agency_certificates(self, request: web.Request) -> web.Response:
        agency_id = request.match_info["agency_id"]

        certificates_info = await self._list_agency_certificates(
            yandex_uid=request["yandex_uid"], agency_id=int(agency_id)
        )

        return json_response(
            data=certificates_schemas.AgencyCertificateSchema(many=True).dump(
                certificates_info
            )
        )

    @docs(
        tags=["certificates"],
        summary="История сертификатов агентства",
        responses={
            200: [certificates_schemas.AgencyCertificatesHistoryEntrySchema(many=True)],
            422: {"name": "Missing data for required field."},
            403: {"description": "Forbidden"},
            404: {"description": "Not found"},
        },
        parameters=[
            {
                "in": "path",
                "name": "agency_id",
                "schema": {"type": "int"},
                "required": "true",
            },
        ],
    )
    @querystring_schema(certificates_schemas.AgencyCertificatesHistoryParamsSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def fetch_agency_certificates_history(
        self, request: web.Request
    ) -> web.Response:
        agency_id = request.match_info["agency_id"]

        params = request['querystring']

        certificates_history_info = await self._fetch_agency_certificates_history(
            yandex_uid=request["yandex_uid"],
            agency_id=int(agency_id),
            project=params.get("project"),
            offset=params["offset"],
            limit=params["limit"],
        )

        return json_response(
            data=certificates_schemas.AgencyCertificatesHistoryEntrySchema(
                many=True
            ).dump(certificates_history_info)
        )

    @docs(
        tags=["certificates"],
        summary="Список сертификатов агентства",
        responses={
            200: [certificates_schemas.AgencyCertificateDetailsSchema()],
            403: {
                "description": "Forbidden",
            },
            404: {
                "description": "Not found",
            },
        },
        parameters=[
            {
                "in": "path",
                "name": "agency_id",
                "schema": {"type": "int"},
                "required": "true",
            },
        ],
    )
    @development_headers_docs
    async def get_agency_certificate_details(
        self, request: web.Request
    ) -> web.Response:
        agency_id = request.match_info["agency_id"]

        details = await self._get_agency_certificate_details(
            yandex_uid=request["yandex_uid"],
            agency_id=int(agency_id),
        )

        return json_response(
            data=certificates_schemas.AgencyCertificateDetailsSchema().dump(details)
        )
