from aiohttp import web
from aiohttp_apispec import docs, querystring_schema
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.handlers.common import (
    development_headers_docs,
    json_response,
)
from crm.agency_cabinet.gateway.server.src.handlers.schemas import acts_schemas, common_documents_schema
from crm.agency_cabinet.gateway.server.src.procedures import acts_procedures

from marshmallow import EXCLUDE


class ActsCollection:
    def __init__(self, sd: ServiceDiscovery):
        self._list_acts = acts_procedures.ListActs(sd)
        self._get_act_url = acts_procedures.GetActUrl(sd)

    @docs(
        tags=['documents'],
        summary='Список актов агентства',
        responses={
            200: {'schema': acts_schemas.ActsListSchema()},
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @querystring_schema(acts_schemas.ListActsParamsSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def list_acts(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        acts = await self._list_acts(
            yandex_uid=request['yandex_uid'],
            agency_id=agency_id,
            **request['querystring']
        )

        data = {
            'items': acts
        }
        return json_response(data=acts_schemas.ActsListSchema().dump(data))

    @docs(
        tags=['documents'],
        summary='Ссылка для скачивания акта',
        responses={
            200: {
                'schema': common_documents_schema.UrlSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            404: {
                'description': 'Not found',
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def get_act_url(self, request: web.Request) -> web.Response:
        act_id = int(request.match_info['act_id'])
        agency_id = int(request.match_info['agency_id'])
        url = await self._get_act_url(request['yandex_uid'], agency_id, act_id)
        return json_response(
            data=common_documents_schema.UrlSchema().dump({'url': url}),
            headers={'Cache-Control': 'no-cache'}
        )
