from aiohttp import web
from aiohttp_apispec import docs, querystring_schema
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.handlers.common import (
    development_headers_docs,
    json_response,
)
from crm.agency_cabinet.gateway.server.src.handlers.schemas import agreements_schemas, common_documents_schema
from crm.agency_cabinet.gateway.server.src.procedures import agreements_procedures

from marshmallow import EXCLUDE


class AgreementsCollection:
    def __init__(self, sd: ServiceDiscovery):
        self._list_agreements = agreements_procedures.ListAgreements(sd)
        self._get_agreement_url = agreements_procedures.GetAgreementUrl(sd)

    @docs(
        tags=['documents'],
        summary='Список дополнительных соглашений агентства',
        responses={
            200: {'schema': agreements_schemas.AgreementsListSchema()},
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @querystring_schema(agreements_schemas.ListAgreementsParamsSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def list_agreements(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        agreements = await self._list_agreements(
            yandex_uid=request['yandex_uid'],
            agency_id=agency_id,
            **request['querystring']
        )

        data = {
            'items': agreements
        }
        return json_response(data=agreements_schemas.AgreementsListSchema().dump(data))

    @docs(
        tags=['documents'],
        summary='Ссылка для скачивания дополнительного соглашения к договору',
        responses={
            200: {
                'schema': common_documents_schema.UrlSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            404: {
                'description': 'Not found',
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def get_agreement_url(self, request: web.Request) -> web.Response:
        agreement_id = int(request.match_info['agreement_id'])
        agency_id = int(request.match_info['agency_id'])
        url = await self._get_agreement_url(
            yandex_uid=request['yandex_uid'],
            agency_id=agency_id,
            agreement_id=agreement_id
        )
        return json_response(
            data=common_documents_schema.UrlSchema().dump({'url': url}),
            headers={'Cache-Control': 'no-cache'}
        )
