from aiohttp import web
from aiohttp_apispec import docs
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.handlers.common import (
    development_headers_docs,
    json_response,
)
from crm.agency_cabinet.gateway.server.src.handlers.schemas import contracts_schemas, common_documents_schema
from crm.agency_cabinet.gateway.server.src.procedures import contracts_procedures


class ContractsCollection:
    def __init__(self, sd: ServiceDiscovery):
        self._list_contracts = contracts_procedures.ListContracts(sd)
        self._get_contract_info = contracts_procedures.GetContractInfo(sd)
        self._get_contract_url = contracts_procedures.GetContractUrl(sd)

    @docs(
        tags=['documents', 'contracts'],
        summary='Список договоров агентства',
        responses={
            200: {"schema": contracts_schemas.ContractsListSchema()},
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def list_contracts(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        contracts = await self._list_contracts(
            yandex_uid=request['yandex_uid'],
            agency_id=agency_id
        )

        data = {
            'items': contracts
        }
        return json_response(data=contracts_schemas.ContractsListSchema().dump(data))

    @docs(
        tags=['documents'],
        summary='Детальная информация о договоре',
        responses={
            200: {
                'schema': contracts_schemas.ContractInfoSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            404: {
                'description': 'Not found',
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def get_contract_info(self, request: web.Request) -> web.Response:
        contract_id = int(request.match_info['contract_id'])
        agency_id = int(request.match_info['agency_id'])
        contract_info = await self._get_contract_info(
            yandex_uid=request['yandex_uid'],
            agency_id=agency_id,
            contract_id=contract_id)
        return json_response(
            data=contracts_schemas.ContractInfoSchema().dump(contract_info)
        )

    @docs(
        tags=['documents'],
        summary='Ссылка для скачивания договора',
        responses={
            200: {
                'schema': common_documents_schema.UrlSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            404: {
                'description': 'Not found',
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def get_contract_url(self, request: web.Request) -> web.Response:
        contract_id = int(request.match_info['contract_id'])
        agency_id = int(request.match_info['agency_id'])
        url = await self._get_contract_url(
            yandex_uid=request['yandex_uid'],
            agency_id=agency_id,
            contract_id=contract_id
        )
        return json_response(
            data=common_documents_schema.UrlSchema().dump({'url': url}),
            headers={'Cache-Control': 'no-cache'}
        )
