from aiohttp import web
from aiohttp_apispec import docs, querystring_schema
from marshmallow import EXCLUDE

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.handlers.common import (
    development_headers_docs,
    json_response,
)
from crm.agency_cabinet.gateway.server.src.handlers.schemas import invoices_schemas, common_documents_schema
from crm.agency_cabinet.gateway.server.src.procedures import invoices_procedures


class InvoicesCollection:
    def __init__(self, sd: ServiceDiscovery):
        self._list_invoices = invoices_procedures.ListInvoices(sd)
        self._get_invoice_info = invoices_procedures.GetInvoiceInfo(sd)
        self._get_invoice_url = invoices_procedures.GetInvoiceUrl(sd)
        self._get_facture_url = invoices_procedures.GetFactureUrl(sd)

    @docs(
        tags=['documents'],
        summary='Список счетов',
        responses={
            200: {
                'schema': invoices_schemas.InvoicesListSchema,
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @querystring_schema(invoices_schemas.GetListInvoicesSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def list_invoices(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        invoices = await self._list_invoices(
            yandex_uid=request['yandex_uid'],
            agency_id=agency_id,
            **request['querystring']
        )

        data = {
            'items': invoices
        }
        return json_response(data=invoices_schemas.InvoicesListSchema().dump(data))

    @docs(
        tags=['documents', 'contracts'],
        summary='Детальная информация о счёте',
        responses={
            200: {
                'schema': invoices_schemas.DetailedInvoiceInfoSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            404: {
                'description': 'Not found',
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def get_invoice_info(self, request: web.Request) -> web.Response:
        invoice_id = int(request.match_info['invoice_id'])
        agency_id = int(request.match_info['agency_id'])
        invoice_info = await self._get_invoice_info(
            yandex_uid=request['yandex_uid'],
            agency_id=agency_id,
            invoice_id=invoice_id)
        return json_response(
            data=invoices_schemas.DetailedInvoiceInfoSchema().dump(invoice_info)
        )

    @docs(
        tags=['documents'],
        summary='Ссылка для скачивания счета',
        responses={
            200: {
                'schema': common_documents_schema.UrlSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            404: {
                'description': 'Not found',
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def get_invoice_url(self, request: web.Request) -> web.Response:
        invoice_id = int(request.match_info['invoice_id'])
        agency_id = int(request.match_info['agency_id'])
        url = await self._get_invoice_url(request['yandex_uid'], agency_id, invoice_id)
        return json_response(
            data=common_documents_schema.UrlSchema().dump({'url': url}),
            headers={'Cache-Control': 'no-cache'}
        )

    @docs(
        tags=['documents'],
        summary='Ссылка для скачивания фактуры',
        responses={
            200: {
                'schema': common_documents_schema.UrlSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            404: {
                'description': 'Not found',
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def get_facture_url(self, request: web.Request) -> web.Response:
        facture_id = int(request.match_info['facture_id'])
        agency_id = int(request.match_info['agency_id'])
        url = await self._get_facture_url(request['yandex_uid'], agency_id, facture_id)
        return json_response(
            data=common_documents_schema.UrlSchema().dump({'url': url}),
            headers={'Cache-Control': 'no-cache'}
        )
