from aiohttp import web
from aiohttp_apispec import docs, json_schema, querystring_schema
from marshmallow import EXCLUDE

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.handlers.common import development_headers_docs, json_response
from crm.agency_cabinet.gateway.server.src.handlers.schemas import ord_acts_schemas, common_schemas
from crm.agency_cabinet.gateway.server.src.procedures import ord_acts_procedures


class OrdActsCollection:
    def __init__(self, sd: ServiceDiscovery):
        self._add_act = ord_acts_procedures.AddAct(sd)
        self._edit_act = ord_acts_procedures.EditAct(sd)
        self._get_acts = ord_acts_procedures.GetActs(sd)

    @docs(
        tags=['ord'],
        summary='Добавление нового акта',
        responses={
            200: {
                'schema': ord_acts_schemas.OrdActSchema()
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @json_schema(ord_acts_schemas.OrdRequestAddOrEditActSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def add_act(self, request: web.Request) -> web.Response:
        report_id = int(request.match_info['report_id'])
        agency_id = int(request.match_info['agency_id'])

        act = await self._add_act(
            agency_id=agency_id,
            yandex_uid=request['yandex_uid'],
            report_id=report_id,
            **request['json']
        )

        return json_response(data=ord_acts_schemas.OrdActSchema().dump(act))

    @docs(
        tags=['ord'],
        summary='Редактирование акта',
        responses={
            200: {
                'schema': common_schemas.SuccessStatusSchema(),
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @json_schema(ord_acts_schemas.OrdRequestAddOrEditActSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def edit_act(self, request: web.Request) -> web.Response:
        report_id = int(request.match_info['report_id'])
        agency_id = int(request.match_info['agency_id'])
        act_id = int(request.match_info['act_id'])

        await self._edit_act(
            agency_id=agency_id,
            yandex_uid=request['yandex_uid'],
            report_id=report_id,
            act_id=act_id,
            **request['json']
        )

        return json_response(data=common_schemas.SuccessStatusSchema().dump({}))

    @docs(
        tags=['ord'],
        summary='Список актов отчета',
        responses={
            200: {
                'schema': ord_acts_schemas.OrdActListSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @querystring_schema(ord_acts_schemas.RequestOrdGetActsSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def get_acts(self, request: web.Request) -> web.Response:
        report_id = int(request.match_info['report_id'])
        agency_id = int(request.match_info['agency_id'])
        data = await self._get_acts(
            yandex_uid=request['yandex_uid'],
            agency_id=agency_id,
            report_id=report_id,
            **request['querystring']
        )
        return json_response(data=ord_acts_schemas.OrdActListSchema().dump(data))
