from aiohttp import web
from aiohttp_apispec import docs, querystring_schema, json_schema
from marshmallow import EXCLUDE

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.handlers.common import development_headers_docs, json_response
from crm.agency_cabinet.gateway.server.src.handlers.schemas import ord_client_rows_schemas, common_schemas
from crm.agency_cabinet.gateway.server.src.procedures import ord_client_rows_procedures


class OrdClientRowsCollection:
    def __init__(self, sd: ServiceDiscovery):
        self._list_client_rows = ord_client_rows_procedures.ListClientRows(sd)
        self._edit_client_row = ord_client_rows_procedures.EditClientRow(sd)

    @docs(
        tags=['ord'],
        summary='Список строк клиента',
        responses={
            200: {
                'schema': ord_client_rows_schemas.OrdClientRowsListSchema()
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @querystring_schema(ord_client_rows_schemas.RequestOrdClientRowsSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def list_client_rows(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        report_id = int(request.match_info['report_id'])
        client_id = int(request.match_info['client_id'])

        result = await self._list_client_rows(
            agency_id=agency_id,
            yandex_uid=request['yandex_uid'],
            report_id=report_id,
            client_id=client_id,
            **request['querystring']
        )

        return json_response(data=ord_client_rows_schemas.OrdClientRowsListSchema().dump(result))

    @docs(
        tags=['ord'],
        summary='Редактирование строки клиента',
        responses={
            200: {
                'schema': common_schemas.SuccessStatusSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @json_schema(ord_client_rows_schemas.OrdEditClientRowSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def edit_client_row(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        report_id = int(request.match_info['report_id'])
        client_id = int(request.match_info['client_id'])
        row_id = int(request.match_info['row_id'])
        await self._edit_client_row(
            agency_id=agency_id,
            yandex_uid=request['yandex_uid'],
            report_id=report_id,
            client_id=client_id,
            row_id=row_id,
            **request['json']
        )
        return json_response(data=common_schemas.SuccessStatusSchema().dump({}))
