from aiohttp import web
from aiohttp_apispec import docs, querystring_schema, json_schema
from marshmallow import EXCLUDE

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.handlers.common import development_headers_docs, json_response
from crm.agency_cabinet.gateway.server.src.handlers.schemas import ord_clients_schemas
from crm.agency_cabinet.gateway.server.src.procedures import ord_clients_procedures


class OrdClientsCollection:
    def __init__(self, sd: ServiceDiscovery):
        self._list_clients = ord_clients_procedures.ListReportClients(sd)
        self._get_client_short_info = ord_clients_procedures.GetClientShortInfo(sd)
        self._create_client = ord_clients_procedures.CreateClient(sd)

    @docs(
        tags=['ord'],
        summary='Список клиентов в отчете',
        responses={
            200: {
                'schema': ord_clients_schemas.OrdReportClientsListSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @querystring_schema(ord_clients_schemas.GetOrdClientsSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def list_clients(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        report_id = int(request.match_info['report_id'])
        clients = await self._list_clients(
            agency_id=agency_id,
            yandex_uid=request['yandex_uid'],
            report_id=report_id,
            **request['querystring']
        )
        data = {
            'size': len(clients),
            'items': clients
        }
        return json_response(data=ord_clients_schemas.OrdReportClientsListSchema().dump(data))

    @docs(
        tags=['ord'],
        summary='Получение сокращенной информции о клиенте',
        responses={
            200: {
                'schema': ord_clients_schemas.OrdClientShortInfoSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            },
            404: {
                'description': 'NotFound'
            }
        }
    )
    @development_headers_docs
    async def get_client_short_info(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        report_id = int(request.match_info['report_id'])
        client_id = int(request.match_info['client_id'])
        client_info = await self._get_client_short_info(
            agency_id=agency_id,
            yandex_uid=request['yandex_uid'],
            report_id=report_id,
            client_id=client_id,
        )
        return json_response(data=ord_clients_schemas.OrdClientShortInfoSchema().dump(client_info))

    @docs(
        tags=['ord'],
        summary='Создание нового клиента',
        responses={
            200: {
                'schema': ord_clients_schemas.OrdReportClientSchema,
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            },
            404: {
                'description': 'NotFound'
            },
            409: {
                'description': 'Conflict'
            }
        }
    )
    @json_schema(ord_clients_schemas.OrdReportCreateClientSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def create_client(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        report_id = int(request.match_info['report_id'])
        client = await self._create_client(
            agency_id=agency_id,
            report_id=report_id,
            yandex_uid=request['yandex_uid'],
            **request['json']
        )
        return json_response(
            data=ord_clients_schemas.OrdReportClientSchema().dump(client))
