from aiohttp import web
from aiohttp_apispec import docs, querystring_schema, json_schema
from marshmallow import EXCLUDE

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.handlers.common import development_headers_docs, json_response
from crm.agency_cabinet.gateway.server.src.handlers.schemas import ord_reports_schemas, common_schemas
from crm.agency_cabinet.gateway.server.src.procedures import ord_reports_procedures, ord_import_data_procedures
from crm.agency_cabinet.gateway.server.src.exceptions import Unprocessable


class OrdReportsCollection:
    MDS_PREFIX = 'ord'

    def __init__(self, sd: ServiceDiscovery):
        self._list_reports = ord_reports_procedures.ListReports(sd)
        self._report_info = ord_reports_procedures.DetailedReportInfo(sd)
        self._send_report = ord_reports_procedures.SendReport(sd)
        self._report_export = ord_reports_procedures.ReportExport(sd)
        self._get_report_export_info = ord_reports_procedures.ReportExportInfo(sd)
        self._report_url = ord_reports_procedures.ReportUrl(sd)
        self._import_data = ord_import_data_procedures.ImportData(sd)
        self._get_lock_status = ord_reports_procedures.GetLockStatus(sd)
        self._delete_report = ord_reports_procedures.DeleteReport(sd)
        self._create_report = ord_reports_procedures.CreateReport(sd)

    @docs(
        tags=['ord'],
        summary='Список отчётов для ОРД',
        responses={
            200: {
                'schema': ord_reports_schemas.OrdReportsListSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @querystring_schema(ord_reports_schemas.RequestOrdReportsInfoSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def list_reports(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        reports = await self._list_reports(agency_id=agency_id, yandex_uid=request['yandex_uid'], **request['querystring'])
        data = {
            'size': len(reports),
            'items': reports
        }
        return json_response(data=ord_reports_schemas.OrdReportsListSchema().dump(data))

    @docs(
        tags=['ord'],
        summary='Подробная информация по отчету ОРД',
        responses={
            200: {
                'schema': ord_reports_schemas.OrdReportSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @development_headers_docs
    async def report_info(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        report_id = int(request.match_info['report_id'])
        data = await self._report_info(agency_id=agency_id, yandex_uid=request['yandex_uid'], report_id=report_id)
        return json_response(data=ord_reports_schemas.OrdReportSchema().dump(data))

    @docs(
        tags=['ord'],
        summary='Отправка отчета в ОРД',
        responses={
            200: {
                'schema': common_schemas.SuccessStatusSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @json_schema(common_schemas.EmptySchema(unknown=EXCLUDE))
    @development_headers_docs
    async def send_report(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        report_id = int(request.match_info['report_id'])
        await self._send_report(agency_id=agency_id, yandex_uid=request['yandex_uid'], report_id=report_id)
        return json_response(data=common_schemas.SuccessStatusSchema().dump({}))

    @docs(
        tags=['ord'],
        summary='Создание задачи на экспорт ОРД отчёта',
        responses={
            200: {
                'schema': ord_reports_schemas.OrdReportExportSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @json_schema(common_schemas.EmptySchema(unknown=EXCLUDE))
    @development_headers_docs
    async def report_export(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        report_id = int(request.match_info['report_id'])
        data = await self._report_export(
            agency_id=agency_id,
            yandex_uid=request['yandex_uid'],
            report_id=report_id
        )
        return json_response(
            data=ord_reports_schemas.OrdReportExportSchema().dump(data),
            headers={'Cache-Control': 'no-cache'}
        )

    @docs(
        tags=['ord'],
        summary='Получение статуса задачи на экспорт ОРД отчета',
        responses={
            200: {
                'schema': ord_reports_schemas.OrdReportExportSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @development_headers_docs
    async def get_report_export_info(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        report_id = int(request.match_info['report_id'])
        report_export_id = int(request.match_info['report_export_id'])
        data = await self._get_report_export_info(
            agency_id=agency_id,
            yandex_uid=request['yandex_uid'],
            report_id=report_id,
            report_export_id=report_export_id
        )
        return json_response(
            data=ord_reports_schemas.OrdReportExportSchema().dump(data),
            headers={'Cache-Control': 'no-cache'}
        )

    @docs(
        tags=['ord'],
        summary='Ссылка на скачивание отчета ОРД',
        responses={
            200: {
                'schema': ord_reports_schemas.OrdReportUrlSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @development_headers_docs
    async def get_report_url(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        report_id = int(request.match_info['report_id'])
        report_export_id = int(request.match_info['report_export_id'])
        url = await self._report_url(
            agency_id=agency_id,
            yandex_uid=request['yandex_uid'],
            report_id=report_id,
            report_export_id=report_export_id
        )
        return json_response(
            data=ord_reports_schemas.OrdReportUrlSchema().dump({'url': url}),
            headers={'Cache-Control': 'no-cache'}
        )

    @docs(
        tags=['ord'],
        summary='Создание таски для импорта отчёта',
        consumes=["multipart/form-data"],
        parameters=[
            {
                "in": "formData",
                "name": "report",
                "type": "file",
                "description": "Отчет для импорта"
            }
        ],
        responses={
            200: {
                'schema': ord_reports_schemas.ReportImportDataSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @development_headers_docs
    async def import_data(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        report_id = int(request.match_info['report_id'])
        data = await request.post()
        filename = None
        content_type = None
        for _, value in data.items():
            data = value.file
            filename = value.filename
            content_type = value.content_type
            break

        if filename is None or content_type is None:
            raise Unprocessable()

        bucket = request.config_dict['mds_bucket']
        s3_resource = request.config_dict['s3_resource']

        result = await self._import_data(agency_id=agency_id, yandex_uid=request['yandex_uid'],
                                         s3_resource=s3_resource, report_id=report_id,
                                         filename=filename, data=data, bucket=bucket, content_type=content_type)
        return json_response(data=ord_reports_schemas.ReportImportDataSchema().dump(result))

    @docs(
        tags=['ord'],
        summary='',
        responses={
            200: {
                'schema': ord_reports_schemas.ReportLockStatusSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @development_headers_docs
    async def get_lock_status(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        report_id = int(request.match_info['report_id'])
        result = await self._get_lock_status(agency_id=agency_id, yandex_uid=request['yandex_uid'], report_id=report_id)
        return json_response(data=ord_reports_schemas.ReportLockStatusSchema().dump(result))

    @docs(
        tags=['ord'],
        summary='Удаление отчета ОРД',
        responses={
            200: {
                'schema': common_schemas.SuccessStatusSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @json_schema(common_schemas.EmptySchema(unknown=EXCLUDE))
    @development_headers_docs
    async def delete_report(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        report_id = int(request.match_info['report_id'])
        await self._delete_report(agency_id=agency_id, yandex_uid=request['yandex_uid'], report_id=report_id)
        return json_response(data=common_schemas.SuccessStatusSchema().dump({}))

    @docs(
        tags=['ord'],
        summary='Создание нового отчета',
        responses={
            200: {
                'schema': ord_reports_schemas.OrdReportSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            }
        }
    )
    @json_schema(ord_reports_schemas.OrdReportCreateSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def create_report(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        report = await self._create_report(
            agency_id=agency_id,
            yandex_uid=request['yandex_uid'],
            **request['json']
        )
        return json_response(data=ord_reports_schemas.OrdReportSchema().dump(report))
