from aiohttp import web
from aiohttp_apispec import docs, querystring_schema
from marshmallow import EXCLUDE

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.procedures import calculator_procedures
from crm.agency_cabinet.gateway.server.src.handlers.common import development_headers_docs, json_response
from crm.agency_cabinet.gateway.server.src.handlers.schemas import calculator_schemas


class CalculatorCollection:
    def __init__(self, sd: ServiceDiscovery):
        self.sd = sd
        self._get_data = calculator_procedures.CalculatorData(sd)
        self._get_meta = calculator_procedures.CalculatorMeta(sd)
        self._list_available_calculators = calculator_procedures.ListAvailableCalculators(sd)

    @staticmethod
    def _get_query_parameters(request):
        data = request['querystring']
        return data['service'], data['contract_id'], data.get('version', '2021')  # TODO: remove 2021

    @docs(
        tags=['calculator'],
        summary='Мета информация для калькулятора',
        responses={
            200: {
                'schema': calculator_schemas.CalculatorMetaSchema,
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            },
            404: {
                'description': 'Contract not found'
            },
            422: {
                'description': 'Unprocessable'
            }

        }
    )
    @querystring_schema(calculator_schemas.CalculatorRequestSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def get_meta(self, request: web.Request) -> web.Response:
        service, contract_id, version = self._get_query_parameters(request)
        yandex_uid = request['yandex_uid']

        agency_id = int(request.match_info['agency_id'])
        schema_object = calculator_schemas.CalculatorMetaSchema()
        meta = await self._get_meta(
            yandex_uid,
            agency_id,
            contract_id,
            service,
            version
        )
        return json_response(data=schema_object.dump(schema_object.loads(meta, unknown=EXCLUDE)))

    @docs(
        tags=['calculator'],
        summary='Исходные данные для калькулятора',
        responses={
            200: {
                'schema': calculator_schemas.CalculatorDataSchema,
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            },
            404: {
                'description': 'Contract not found'
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @querystring_schema(calculator_schemas.CalculatorRequestSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def get_init_data(self, request: web.Request) -> web.Response:
        service, contract_id, version = self._get_query_parameters(request)
        agency_id = int(request.match_info['agency_id'])
        yandex_uid = request['yandex_uid']
        data = await self._get_data(yandex_uid, agency_id, contract_id, service, version)
        schema = calculator_schemas.CalculatorDataSchema
        return json_response(data=schema().dump(schema().loads(data, unknown=EXCLUDE)))

    @docs(
        tags=['calculator'],
        summary='Список доступных калькуляторов',
        responses={
            200: {
                'schema': calculator_schemas.CalculatorsDescriptionsSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @querystring_schema(calculator_schemas.AvailableCalculatorsRequestSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def list_available_calculators(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        yandex_uid = request['yandex_uid']
        data = await self._list_available_calculators(yandex_uid, agency_id, **request['querystring'])
        return json_response(data=calculator_schemas.CalculatorsDescriptionsSchema().dump(data))
