import logging
from aiohttp import web
from aiohttp_apispec import docs, json_schema, querystring_schema
from marshmallow import EXCLUDE

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.exceptions import InternalServerError
from crm.agency_cabinet.gateway.server.src.handlers.common import development_headers_docs, json_response
from crm.agency_cabinet.gateway.server.src.handlers.schemas import reports_schemas, common_schemas
from crm.agency_cabinet.gateway.server.src.procedures import reports_procedures


LOGGER = logging.getLogger('handlers.reports')


class ReportsCollection:
    def __init__(self, sd: ServiceDiscovery):
        self._list_reports = reports_procedures.ListReports(sd)
        self._report_info = reports_procedures.DetailedReportInfo(sd)
        self._delete_report = reports_procedures.DeleteReport(sd)
        self._create_report = reports_procedures.CreateReport(sd)
        self._get_report_url = reports_procedures.GetReportUrl(sd)

    @docs(
        tags=['reports'],
        summary='Список отчетов',
        responses={
            200: {
                'schema': reports_schemas.ReportsListSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @querystring_schema(reports_schemas.GetReportsInfoSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def list_reports(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        reports = await self._list_reports(request['yandex_uid'], agency_id, **request['querystring'])
        data = {
            'reports': reports
        }
        return json_response(data=reports_schemas.ReportsListSchema().dump(data))

    @docs(
        tags=['reports'],
        summary='Детальная информация об отчете',
        responses={
            200: {
                'schema': reports_schemas.ReportInfoSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            404: {
                'description': 'Not found',
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def report_info(self, request: web.Request) -> web.Response:
        report_id = int(request.match_info['report_id'])
        agency_id = int(request.match_info['agency_id'])
        report = await self._report_info(request['yandex_uid'], agency_id, report_id)
        return json_response(data=reports_schemas.ReportInfoSchema().dump(report))

    @docs(
        tags=['reports'],
        summary='Создание отчета',
        responses={
            200: {
                'schema': reports_schemas.ReportInfoSchema(),
                'description': 'Ok'
            },
            400: {
                'description': 'Bad request',
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @json_schema(reports_schemas.CreateReportInfoSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def create_report(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        result = await self._create_report(request['yandex_uid'], agency_id, request['json'])
        return json_response(data=reports_schemas.ReportInfoSchema().dump(result))

    @docs(
        tags=['reports'],
        summary='Удаление отчета',
        responses={
            200: {
                'description': 'Ok',
                'schema': common_schemas.DeleteStatusSchema()
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            404: {
                'description': 'Not found',
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @json_schema(common_schemas.EmptySchema())
    @development_headers_docs
    async def delete_report(self, request: web.Request) -> web.Response:
        report_id = int(request.match_info['report_id'])
        agency_id = int(request.match_info['agency_id'])
        result = await self._delete_report(request['yandex_uid'], agency_id, report_id)
        if result:
            return json_response(data=common_schemas.DeleteStatusSchema().dump({}))
        raise InternalServerError('Unsuccessful report delete operation')

    @docs(
        tags=['reports'],
        summary='Ссылка для скачивания отчета',
        responses={
            200: {
                'schema': reports_schemas.ReportUrlSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            404: {
                'description': 'Not found',
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def report_url(self, request: web.Request) -> web.Response:
        report_id = int(request.match_info['report_id'])
        agency_id = int(request.match_info['agency_id'])
        url = await self._get_report_url(request['yandex_uid'], agency_id, report_id)
        return json_response(data=reports_schemas.ReportUrlSchema().dump({'url': url}),
                             headers={'Cache-Control': 'no-cache'})
