from aiohttp import web
from aiohttp_apispec import docs, querystring_schema
from marshmallow import EXCLUDE

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.handlers.common import development_headers_docs, json_response
from crm.agency_cabinet.gateway.server.src.handlers.schemas import rewards_schemas
from crm.agency_cabinet.gateway.server.src.handlers.schemas.dashboard import DashboardSchema, GetDashboardSchema
from crm.agency_cabinet.gateway.server.src.procedures import rewards_procedures


class RewardsCollection:
    def __init__(self, sd: ServiceDiscovery):
        self._list_rewards = rewards_procedures.ListRewards(sd)
        self._reward_info = rewards_procedures.DetailedRewardInfo(sd)
        self._get_dashboard = rewards_procedures.GetDashboard(sd)

    @docs(
        tags=['rewards'],
        summary='Список агентских премий',
        responses={
            200: {
                'schema': rewards_schemas.RewardsListSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @querystring_schema(rewards_schemas.GetRewardsMinimalInfoSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def list_rewards(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        rewards = await self._list_rewards(request['yandex_uid'], agency_id, **request['querystring'])
        data = {
            'rewards': rewards
        }
        return json_response(data=rewards_schemas.RewardsListSchema().dump(data))

    @docs(
        tags=['rewards'],
        summary='Информация о премии',
        responses={
            200: {
                'schema': rewards_schemas.DetailedRewardInfoSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            404: {
                'description': 'Bad request',
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def reward_info(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        reward_id = int(request.match_info['reward_id'])

        reward = await self._reward_info(request['yandex_uid'], agency_id, reward_id)
        return json_response(data=rewards_schemas.DetailedRewardInfoSchema().dump(reward))

    @docs(
        tags=['rewards'],
        summary='Премии в разбивке на контракты, сервисы и периоды',
        responses={
            200: {
                'schema': DashboardSchema,
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @querystring_schema(GetDashboardSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def dashboard(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        dashboard = await self._get_dashboard(request['yandex_uid'], agency_id, **request['querystring'])

        return json_response(data=DashboardSchema().dump({'dashboard': dashboard}))
