from aiohttp import web
from aiohttp_apispec import docs, querystring_schema, json_schema
from marshmallow import EXCLUDE

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.exceptions import InternalServerError
from crm.agency_cabinet.gateway.server.src.handlers.common import development_headers_docs, json_response
from crm.agency_cabinet.gateway.server.src.handlers.schemas import roles_schemas
from crm.agency_cabinet.gateway.server.src.procedures import roles_procedures


class RoleModelCollection:
    def __init__(self, sd: ServiceDiscovery):
        self._list_users = roles_procedures.ListPermissions(sd)
        self._edit_user = roles_procedures.EditUser(sd)
        self._list_suggest_users = roles_procedures.ListSuggestUsers(sd)
        self._list_roles = roles_procedures.ListRoles(sd)
        self._list_user_roles = roles_procedures.ListUserRoles(sd)
        self._list_available_partners = roles_procedures.ListPartners(sd)

    @docs(
        tags=['roles'],
        summary='Список сотрудников агенства, у которых есть доступ '
                'к агенсткому кабинету + инфа про их роли и пермишены',
        responses={
            200: {
                'schema': roles_schemas.UsersListSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @querystring_schema(roles_schemas.GetUsersInfoSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def list_users(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])

        users_list = []
        for user in await self._list_users(request['yandex_uid'], agency_id, **request['querystring']):
            r = {
                'user_id': user.user.user_id,
                'current': user.user.current,
                'name': user.user.name,
                'email': user.user.email,
                'login': user.user.login,
                'roles': [
                    {
                        'role_id': role_permission.role.role_id,
                        'name': role_permission.role.name,
                        'display_name': role_permission.role.display_name,
                        'permissions': [
                            {
                                'name': permission.name
                            } for permission in role_permission.permissions
                        ]
                    } for role_permission in user.roles_permissions
                ]  # TODO: use pluck/move to pre_dump
            }
            if user.user.avatar_id:
                r['avatar_id'] = user.user.avatar_id
            users_list.append(r)

        data = {
            'users': users_list
        }

        return json_response(data=roles_schemas.UsersListSchema().dump(data))

    @docs(
        tags=['roles'],
        summary='Управление ролями и пермишенами сотрудника',
        responses={
            200: {
                'description': 'Ok',
                'schema': roles_schemas.EditPermissionsStatusSchema()
            },
            400: {
                'description': 'Bad request',
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @json_schema(roles_schemas.InputRoleSchema(many=True, unknown=EXCLUDE))
    @development_headers_docs
    async def edit_user(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])
        user_id = int(request.match_info['user_id'])
        roles = request['json']
        result = await self._edit_user(request['yandex_uid'], agency_id, user_id, roles)
        if result:
            return json_response(data=roles_schemas.EditPermissionsStatusSchema().dump({}))

        raise InternalServerError('Unsuccessful user role editing operation')

    @docs(
        tags=['roles'],
        summary='Список сотрудников агентства, у которых нет доступа к агенсткому кабинету',
        responses={
            200: {
                'schema': roles_schemas.SuggestUsersListSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @querystring_schema(roles_schemas.GetSuggestedUsersInfoSchema(unknown=EXCLUDE))
    @development_headers_docs
    async def list_suggest_users(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])

        data = {
            'users': await self._list_suggest_users(request['yandex_uid'], agency_id, **request['querystring'])
        }
        return json_response(data=roles_schemas.SuggestUsersListSchema().dump(data))

    @docs(
        tags=['roles'],
        summary='Информация про роли и пермишены (какие бывают роли, какие пермишены '
                'для них определены, можно ли редактировать пермишены)',
        responses={
            200: {
                'schema': roles_schemas.RoleInfoListSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def list_roles(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])

        roles = await self._list_roles(request['yandex_uid'], agency_id)
        data = {
            'roles': [
                {
                    'role_id': role_permissions.role.role_id,
                    'name': role_permissions.role.name,
                    'display_name': role_permissions.role.display_name,
                    'permissions': [
                        {
                            'name': permission.name,
                            'editable': permission.is_editable,
                        }
                        for permission in role_permissions.permissions
                    ]
                }
                for role_permissions in roles
            ]
        }
        return json_response(data=roles_schemas.RoleInfoListSchema().dump(data))

    @docs(
        tags=['roles'],
        summary='Список ролей и пермишенов для текущего пользователя',
        responses={
            200: {
                'schema': roles_schemas.RoleListSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def list_user_roles(self, request: web.Request) -> web.Response:
        agency_id = int(request.match_info['agency_id'])

        roles = await self._list_user_roles(request['yandex_uid'], agency_id)
        data = {
            'roles': [
                {
                    'role_id': role_permissions.role.role_id,
                    'name': role_permissions.role.name,
                    'display_name': role_permissions.role.display_name,
                    'permissions': [
                        {
                            'name': permission.name,
                        }
                        for permission in role_permissions.permissions
                    ]
                }
                for role_permissions in roles
            ]
        }
        return json_response(data=roles_schemas.RoleListSchema().dump(data))

    @docs(
        tags=['roles'],
        summary='Список партнеров',
        responses={
            200: {
                'schema': roles_schemas.PartnersListSchema(),
                'description': 'Ok'
            },
            401: {
                'description': 'Unauthorized'
            },
            403: {
                'description': 'Forbidden'  # TODO: у ошибок есть схема, поэтому надо добавить в спеку
            },
            422: {
                'description': 'Unprocessable'
            }
        }
    )
    @development_headers_docs
    async def list_available_partners(self, request: web.Request) -> web.Response:
        partners = await self._list_available_partners(request['yandex_uid'])
        data = {
            'partners': partners,
            'size': len(partners)  # TODO: get size from microservice
        }
        return json_response(data=roles_schemas.PartnersListSchema().dump(data))
