from crm.agency_cabinet.common.consts import AverageCheckBucket
from marshmallow import Schema, fields, validate, validates_schema, ValidationError
from decimal import Decimal


class MarketDatasetSchema(Schema):
    average_budget = fields.Float()
    customers = fields.Integer()  # or clients?


class IncreaseDatasetSchema(Schema):
    new_customers = fields.Integer()
    new_customers_prev = fields.Integer()
    customers = fields.Integer()
    increase_percent = fields.Method(dump_only=True, serialize="compute_increase_percent")

    def compute_increase_percent(self, obj):
        try:
            if hasattr(obj, 'new_customers'):
                l = obj.new_customers
            else:
                l = obj['new_customers']
            if hasattr(obj, 'new_customers_prev'):
                r = obj.new_customers_prev
            else:
                r = obj['new_customers_prev']
            if hasattr(obj, 'customers'):
                fallback_divisor = obj.customers
            else:
                fallback_divisor = obj['customers']
        except (KeyError, AttributeError, TypeError) as ex:
            raise ValidationError('Invalid object for dump: don\'t have field {}'.format(str(ex)))
        if not isinstance(l, int) or not isinstance(r, int):
            raise ValidationError('Invalid type for field')
        # TODO: fix logic with percent increase if r = 0
        return float((Decimal(r - l) / Decimal(r)) * Decimal(100) if r else Decimal(l) / max(1, Decimal(fallback_divisor)) * Decimal(100))


class MarketPieDatasetSchema(Schema):
    percent = fields.Float()
    grade = fields.String(validate=validate.OneOf([e.value for e in AverageCheckBucket]))  # AverageCheckBucket
    customers = fields.Integer()


class IncreaseBarDatasetSchema(Schema):
    customers_at_left_date = fields.Integer()
    customers_at_right_date = fields.Integer()
    increase = fields.Method(dump_only=True, serialize="compute_increase")

    def compute_increase(self, obj):
        # TODO: использовать pre-process тк swagger не умеет рисовать такие поля
        try:
            if hasattr(obj, 'customers_at_left_date'):
                l = obj.customers_at_left_date
            else:
                l = obj['customers_at_left_date']
            if hasattr(obj, 'customers_at_right_date'):
                r = obj.customers_at_right_date
            else:
                r = obj['customers_at_right_date']
        except (KeyError, AttributeError, TypeError) as ex:
            raise ValidationError('Invalid object for dump: don\'t have field {}'.format(str(ex)))
        if not isinstance(l, int) or not isinstance(r, int):
            raise ValidationError('Invalid type for field "increase"')
        return r - l


class MarketSituationGraphDescriptionSchema(Schema):
    agencies_with_less_average_budget_percent = fields.Float()


class MarketSituationGraphSchema(Schema):
    other = fields.List(fields.Nested(MarketDatasetSchema))
    current_at_left_date = fields.Nested(MarketDatasetSchema)
    current_at_right_date = fields.Nested(MarketDatasetSchema)
    description = fields.Nested(MarketSituationGraphDescriptionSchema)


class IncreaseGraphDescriptionSchema(Schema):
    increase_for_period = fields.Float()
    percentage_of_agencies_with_less_increase = fields.Float()


class IncreaseGraphSchema(Schema):
    other = fields.List(fields.Nested(IncreaseDatasetSchema))
    current_at_left_date = fields.Nested(IncreaseDatasetSchema)
    current_at_right_date = fields.Nested(IncreaseDatasetSchema)
    description = fields.Nested(IncreaseGraphDescriptionSchema)


class ClientByGradesGraphDescriptionSchema(Schema):
    average_budget = fields.Float()
    is_higher_than_other_agencies = fields.Bool()
    most_customers_grade = fields.String(validate=validate.OneOf([e.value for e in AverageCheckBucket]))


class ClientByGradesGraphSchema(Schema):
    other = fields.List(fields.Nested(MarketPieDatasetSchema))
    current = fields.List(fields.Nested(MarketPieDatasetSchema))
    description = fields.Nested(ClientByGradesGraphDescriptionSchema)


class ActiveClientsGraphDescriptionSchema(Schema):
    percentage_of_agencies_with_less_clients = fields.Float()


class ActiveClientsGraphSchema(Schema):
    other = fields.List(fields.Nested(IncreaseBarDatasetSchema))
    current = fields.Nested(IncreaseBarDatasetSchema)
    description = fields.Nested(ActiveClientsGraphDescriptionSchema)


class RequestAnalyticsSchema(Schema):
    left_date = fields.Date(required=True)
    right_date = fields.Date(required=True)

    @validates_schema
    def _check_date(self, data, *args, **kwargs):
        if data['left_date'].day != 1 or data['right_date'].day != 1:
            raise ValidationError("Only start of month supported")
        if data["left_date"] >= data["right_date"]:
            raise ValidationError(
                "Right date must be greater then left date."
            )
