from marshmallow import (
    Schema,
    ValidationError,
    fields,
    post_dump,
    validate,
    validates_schema,
)
from marshmallow_enum import EnumField

from crm.agency_cabinet.common.consts.report import ReportsStatuses
from crm.agency_cabinet.client_bonuses.common.structs import BonusStatusType, BonusType, ClientType


class ClientBonusSchema(Schema):
    client_id = fields.String(required=True)
    email = fields.String(required=True)
    accrued = fields.String(required=True, data_key="accumulated", allow_none=True)
    spent = fields.String(required=True, allow_none=True)
    awarded = fields.String(required=True, allow_none=True)
    active = fields.Bool(required=True)
    currency = fields.String(required=True)


class ClientBonusSettingsSchema(Schema):
    first_date = fields.AwareDateTime(format="iso")
    last_date = fields.AwareDateTime(format="iso")


class ListClientBonusesParamsSchema(Schema):
    client_type = EnumField(ClientType, required=True)
    bonus_type = EnumField(BonusType, required=True)
    limit = fields.Integer(required=True, validate=validate.Range(min=1, max=100))
    offset = fields.Integer(required=True, validate=validate.Range(min=0))
    datetime_start = fields.AwareDateTime(required=True, format="iso")
    datetime_end = fields.AwareDateTime(required=True, format="iso")
    search_query = fields.String(validate=validate.Length(min=1))

    @validates_schema
    def _check_params(self, data, *args, **kwargs):
        if data["datetime_end"] < data["datetime_start"]:
            raise ValidationError(
                "Start datetime must be grater or equal end datetime."
            )


class FetchClientBonusesDetailsParamsSchema(Schema):
    datetime_start = fields.AwareDateTime(required=True, format="iso")
    datetime_end = fields.AwareDateTime(required=True, format="iso")

    @validates_schema
    def _check_params(self, data, *args, **kwargs):
        if data["datetime_end"] < data["datetime_start"]:
            raise ValidationError(
                "Start datetime must be grater or equal end datetime."
            )


class BonusAmountSchema(Schema):
    program_id = fields.Integer(required=True)
    amount = fields.String(required=True)


class ClientBonusDetailSchema(Schema):
    type = EnumField(BonusStatusType, required=True)
    amounts = fields.Nested(BonusAmountSchema, required=True, many=True)
    total = fields.String(required=True)
    date = fields.AwareDateTime(required=True, format="iso")


class GraphPointSchema(Schema):
    point = fields.AwareDateTime(required=True, format="iso")
    value = fields.String(required=True)

    @post_dump(pass_many=True)
    def format_graph_points(self, data, **kwargs):
        return {point["point"]: point["value"] for point in data}


class ClientBonusesGraphProgramsSchema(Schema):
    program_id = fields.Integer(required=True)
    historical_monthly_data = fields.Nested(GraphPointSchema, required=True, many=True)


class ClientBonusesGraphSchema(Schema):
    bonuses_available = fields.String(required=True)
    overall_spent = fields.Nested(GraphPointSchema, required=True, many=True)
    overall_accrued = fields.Nested(GraphPointSchema, required=True, many=True)
    programs = fields.Nested(ClientBonusesGraphProgramsSchema, required=True, many=True)


class ClientBonusesReportUrlSchema(Schema):
    url = fields.URL()


class ClientBonusesReportInfoSchema(Schema):
    id = fields.Integer(required=True)
    name = fields.String(required=True)
    created_at = fields.AwareDateTime(format="iso")
    period_from = fields.AwareDateTime(format="iso", required=True)
    period_to = fields.AwareDateTime(format="iso")
    status = fields.String(validate=validate.OneOf({e.value for e in ReportsStatuses}))
    client_type = EnumField(ClientType, required=True)


class CreateBonusesReportInfoSchema(ClientBonusesReportInfoSchema):
    class Meta:
        exclude = ['id', 'status', 'created_at']


class CashbackProgramSchema(Schema):
    id = fields.Integer(required=True)
    category_id = fields.Integer(required=True)
    is_general = fields.Boolean(required=True)
    is_enabled = fields.Boolean(required=True)
    name_ru = fields.String(required=True)
    name_en = fields.String(required=True)
    description_ru = fields.String(required=True)
    description_en = fields.String(required=True)


class CashbackProgramsListSchema(Schema):
    programs = fields.List(fields.Nested(CashbackProgramSchema))
