from marshmallow import Schema, fields, validate
from marshmallow_enum import EnumField

from crm.agency_cabinet.certificates.common.structs import (
    EmployeeCertificateStatus,
)

__all__ = ["CertifiedEmployeeSchema", "AgencyCertificateSchema"]


class EmployeeCertificateSchema(Schema):
    project = fields.String(required=True)
    start_time = fields.AwareDateTime(required=True, format="iso")
    expiration_time = fields.AwareDateTime(required=True, format="iso")
    external_id = fields.String(required=True)
    status = EnumField(
        enum=EmployeeCertificateStatus, required=True, by_value=True
    )


class CertifiedEmployeeSchema(Schema):
    name = fields.String(allow_none=True)
    email = fields.String(required=True)
    agency_id = fields.Integer(required=True)
    certificates = fields.Nested(EmployeeCertificateSchema, many=True)


class AgencyCertificateSchema(Schema):
    id = fields.Integer(required=True)
    project = fields.String(required=True)
    expiration_time = fields.AwareDateTime(required=True, format="iso")
    auto_renewal_is_met = fields.Boolean(required=True)


class AgencyCertificatesHistoryEntrySchema(Schema):
    id = fields.Integer(required=True, data_key="certificate_id")
    project = fields.String(required=True)
    start_time = fields.AwareDateTime(required=True, format="iso")
    expiration_time = fields.AwareDateTime(required=True, format="iso")


class PaginationSchema(Schema):
    limit = fields.Integer(required=True, validate=validate.Range(min=1, max=100))
    offset = fields.Integer(required=True, validate=validate.Range(min=0))


class AgencyCertificatesHistoryParamsSchema(PaginationSchema):
    project = fields.String(validate=validate.Length(min=1))


class EmployeeCertificatesParamsSchema(PaginationSchema):
    project = fields.String(validate=validate.Length(min=1))
    search_query = fields.String(validate=validate.Length(min=1))
    status = fields.String(validate=validate.OneOf([status.value for status in EmployeeCertificateStatus]))


class DirectKPI(Schema):
    name = fields.String(required=True)
    group = fields.String(required=True)
    value = fields.Float(required=True)
    max_value = fields.Float(required=True)


class DirectBonusPoint(Schema):
    name = fields.String(required=True)
    value = fields.String(required=True)
    threshold = fields.String(required=True)
    score = fields.Float(required=True)
    is_met = fields.Boolean(required=True)


class DirectCertificationCondition(Schema):
    name = fields.String(required=True)
    value = fields.String(required=True)
    threshold = fields.String(required=True)
    is_met = fields.Boolean(required=True)


class DirectCertificationScore(Schema):
    score_group = fields.String(required=True)
    value = fields.Float(required=True)
    threshold = fields.Float(required=True)
    is_met = fields.Boolean(required=True)


class AgencyCertificateDetailsSchema(Schema):
    agency_id = fields.Integer(required=True)
    conditions = fields.Nested(DirectCertificationCondition, required=True, many=True)
    kpis = fields.Nested(DirectKPI, required=True, many=True)
    bonus_points = fields.Nested(DirectBonusPoint, required=True, many=True)
    scores = fields.Nested(DirectCertificationScore, required=True, many=True)
