from datetime import date
from dateutil.relativedelta import relativedelta
from marshmallow import fields, Schema, validate, validates_schema, ValidationError, post_load

from crm.agency_cabinet.common.consts import ReportsStatuses, ReportsTypes, Services

__all__ = [
    'ReportInfoSchema',
    'ReportsListSchema',
    'GetReportsInfoSchema',
    'CreateReportInfoSchema'
]


class ReportInfoSchema(Schema):
    id = fields.Integer()
    contract_id = fields.Integer(required=True)
    name = fields.String(required=True)
    service = fields.String(validate=validate.OneOf({e.value for e in Services}), required=True)
    type = fields.String(validate=validate.OneOf({e.value for e in ReportsTypes}), required=True)
    created_at = fields.DateTime()
    period_from = fields.Date(required=True)
    period_to = fields.Date()
    status = fields.String(validate=validate.OneOf({e.value for e in ReportsStatuses}))
    clients = fields.List(fields.Integer())
    # TODO: validate clients


class CreateReportInfoSchema(ReportInfoSchema):
    class Meta:
        exclude = ['id', 'status', 'created_at']

    @validates_schema
    def validate_period_to(self, data, **kwargs):
        type = data.get('type')
        if type == ReportsTypes.custom.value:
            raise ValidationError('Not supported now')
        # period_to = data.get('period_to')
        # period_from = data.get('period_from')
        # if type == ReportsTypes.custom.value and period_to is None:
        #    raise ValidationError('Period_to required for custom report type')
        # elif type == ReportsTypes.month.value \
        #         and period_to is not None and period_from + relativedelta(months=1) != period_to:
        #     raise ValidationError('Custom period_to should\'t be specified for month report type')
        # elif type == ReportsTypes.quarter.value \
        #         and period_to is not None and period_from + relativedelta(months=3) != period_to:
        #     raise ValidationError('Custom period_to should\'t be specified for quarter report type')

    @post_load
    def compute_period_to(self, data, **kwargs):
        period_to = data.get('period_to')
        type = data.get('type')
        # if period_to is None:  # TODO: return check after removing period_to from frontend query
        period_from: date = data['period_from']
        if type == ReportsTypes.month.value:
            period_to = period_from + relativedelta(months=1) - relativedelta(days=1)  # последний день месяца
        elif type in (ReportsTypes.quarter_direct.value, ReportsTypes.quarter_video.value):
            period_to = period_from + relativedelta(months=3) - relativedelta(days=1)  # последний день месяца
        data['period_to'] = period_to
        return data


class ReportsListSchema(Schema):
    reports = fields.List(fields.Nested(ReportInfoSchema))


class ReportUrlSchema(Schema):
    url = fields.URL()


class GetReportsInfoSchema(Schema):
    contract_id = fields.Integer(
        description="Если указано, то возвращаются только те отчеты, что привязаны к заданному контракту")
    type = fields.String(
        validate=validate.OneOf({e.value for e in ReportsTypes}),
        description="Если указано, то возвращаются только те отчеты, у которых тип совпадает с заданным")
    service = fields.String(
        validate=validate.OneOf({e.value for e in Services}),
        description="Если указано, то возвращаются только те отчеты, чей сервис совпадает с заданным")
