import typing
import logging
import re
from aiohttp import web
from aiohttp_apispec import validation_middleware
from aiohttp_apispec.utils import issubclass_py37fix
from dataclasses import dataclass
from aiohttp.web_middlewares import middleware
from aiohttp.web_request import Request
from aiohttp.web_response import Response
from crm.agency_cabinet.gateway.server.src.exceptions import BadRequest, InternalServerError

MAP_CONTENT_TYPE_TO_REGEXP = {
    'application/json': re.compile(r"^application/(?:[\w.+-]+?\+)?json")
}

LOGGER = logging.getLogger('middlewares.validation')


@dataclass
@middleware
class ValidationMiddleware:

    async def __call__(self, request: Request, handler: typing.Callable[[Request], typing.Awaitable[Response]]) -> Response:
        if request.method in ('POST', 'PUT', 'PATCH'):
            orig_handler = request.match_info.handler
            if not hasattr(orig_handler, "__apispec__"):
                if not issubclass_py37fix(orig_handler, web.View):
                    raise InternalServerError('Can\'t handle request')
                sub_handler = getattr(orig_handler, request.method.lower(), None)
                if sub_handler is None:
                    raise InternalServerError('Can\'t handle request')
                if not hasattr(sub_handler, "__apispec__"):
                    raise InternalServerError('Bad handler')
                apispec = sub_handler.__apispec__
            else:
                apispec = orig_handler.__apispec__

            consumes = set(apispec.get('consumes') or ['application/json'])
            accepted = False
            for accepting_type in consumes:
                accepting_type = MAP_CONTENT_TYPE_TO_REGEXP.get(accepting_type, accepting_type)
                if isinstance(accepting_type, str):
                    accepted = accepting_type == request.content_type
                elif isinstance(accepting_type, typing.Pattern):
                    accepted = accepting_type.match(request.content_type)
                if accepted:
                    break

            if not accepted:
                raise BadRequest(f'Incorrect mime-type: {request.content_type}')

        return await validation_middleware(request, handler)
