from dataclasses import dataclass

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.agencies.common import structs as agencies_structs
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound
from crm.agency_cabinet.grants.common import structs as grants_structs
from crm.agency_cabinet.grants.client import NoSuchUserException
from crm.agency_cabinet.rewards.common import structs as rewards_structs
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants, check_feature


@dataclass
class AgencyInfoWithContracts(agencies_structs.AgencyInfo):
    contracts: list


@dataclass
class ListAgencies:
    __slots__ = "sd"

    sd: ServiceDiscovery

    async def __call__(self, yandex_uid: int) -> list[agencies_structs.AgencyInfo]:
        # TODO: temp, delete
        if check_feature('AGENCIES_API_NEW_GRANTS') or check_feature('ANY_API_NEW_GRANTS'):
            try:
                agencies_access = await self.sd.grants.get_accessible_agencies(yandex_uid)
            except NoSuchUserException:
                # raise AccessDenied(e.message)
                agencies_access = []
        else:
            access_scope_type, agencies_access = await self.sd.grants.get_access_scope(yandex_uid)
            if access_scope_type == grants_structs.AccessScopeType.ALL:
                return await self.sd.agencies.get_all_agencies_info()

        if len(agencies_access) == 0:
            return []

        return await self.sd.agencies.get_agencies_info(agencies_access)


@dataclass
class AgencyInfo:
    __slots__ = "sd"

    sd: ServiceDiscovery

    @check_grants(feature_flag_name='AGENCIES_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int) -> AgencyInfoWithContracts:

        agency_info = await self.sd.agencies.get_agencies_info([agency_id])
        if len(agency_info) == 0:
            raise NotFound()

        agency_info = agency_info[0]
        return AgencyInfoWithContracts(
            agency_id=agency_info.agency_id,
            name=agency_info.name,
            phone=agency_info.phone,
            email=agency_info.email,
            site=agency_info.site,
            actual_address=agency_info.actual_address,
            legal_address=agency_info.legal_address,
            contracts=[]
        )


@dataclass
class ListContracts:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(feature_flag_name='AGENCIES_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int) -> list[rewards_structs.ContractInfo]:
        return await self.sd.rewards.get_contracts_info(agency_id)


@dataclass
class ListClients:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(feature_flag_name='AGENCIES_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int,
                       limit: int = None, offset: int = None, search_query: str = None) -> list[agencies_structs.ClientInfo]:
        return await self.sd.agencies.get_clients_info(agency_id, limit=limit, offset=offset, search_query=search_query)
