from dataclasses import dataclass
from datetime import datetime
from typing import Optional

from crm.agency_cabinet.client_bonuses.client import ClientNotFound, NoSuchReportException, UnsuitableAgencyException, \
    FileNotFoundException, ReportNotReadyException
from crm.agency_cabinet.client_bonuses.common.structs import (
    BonusDetails,
    BonusType,
    ClientBonus,
    ClientGraph,
    ClientType,
    ClientBonusSettings,
    ReportInfo,
    CashbackProgram,
)
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants
from crm.agency_cabinet.grants.common.consts import Permissions

__all__ = ["ListBonuses", "GetBonusesSettings", "FetchClientBonusesDetails", "FetchClientBonusesGraph",
           "GetClientBonusesReportUrl", "ListBonusesReports", "CreateReport", "DeleteReport",
           "ListCashbackPrograms"]


@dataclass
class ListBonuses:
    __slots__ = "sd"

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.client_bonuses.value,), feature_flag_name='BONUSES_API_NEW_GRANTS')
    async def __call__(
        self,
        yandex_uid: int,
        agency_id: int,
        limit: int,
        offset: int,
        client_type: ClientType,
        bonus_type: BonusType,
        datetime_start: datetime,
        datetime_end: datetime,
        search_query: Optional[str],
    ) -> list[ClientBonus]:
        return await self.sd.client_bonuses.list_clients_bonuses(
            agency_id=agency_id,
            limit=limit,
            client_type=client_type,
            bonus_type=bonus_type,
            offset=offset,
            datetime_start=datetime_start,
            datetime_end=datetime_end,
            search_query=search_query
        )


@dataclass
class GetBonusesSettings:
    __slots__ = "sd"

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.client_bonuses.value,), feature_flag_name='BONUSES_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int) -> ClientBonusSettings:

        return await self.sd.client_bonuses.get_clients_bonuses_settings(
            agency_id=agency_id,
        )


@dataclass
class FetchClientBonusesDetails:
    __slots__ = "sd"

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.client_bonuses.value,), feature_flag_name='BONUSES_API_NEW_GRANTS')
    async def __call__(
        self,
        yandex_uid: int,
        agency_id: int,
        client_id: int,
        datetime_start: datetime,
        datetime_end: datetime,
    ) -> list[BonusDetails]:

        try:
            return await self.sd.client_bonuses.fetch_bonuses_details(
                agency_id=agency_id,
                client_id=client_id,
                datetime_start=datetime_start,
                datetime_end=datetime_end,
            )
        except ClientNotFound:
            raise NotFound()


@dataclass
class FetchClientBonusesGraph:
    __slots__ = "sd"

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.client_bonuses.value,), feature_flag_name='BONUSES_API_NEW_GRANTS')
    async def __call__(
        self,
        yandex_uid: int,
        agency_id: int,
        client_id: int,
    ) -> ClientGraph:

        try:
            return await self.sd.client_bonuses.fetch_client_graph(
                agency_id=agency_id,
                client_id=client_id,
            )
        except ClientNotFound:
            raise NotFound()


@dataclass
class ListBonusesReports:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.client_bonuses.value,), feature_flag_name='BONUSES_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int) -> list[ReportInfo]:

        reports_info = await self.sd.client_bonuses.list_bonuses_reports(agency_id=agency_id)
        return reports_info


@dataclass
class GetClientBonusesReportUrl:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.client_bonuses.value,), feature_flag_name='BONUSES_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int, report_id: int) -> str:
        try:
            url = await self.sd.client_bonuses.get_report_url(agency_id, report_id)
            return url
        except NoSuchReportException:
            raise NotFound(f'Can\'t find report with id {report_id}')
        except UnsuitableAgencyException:
            raise NotFound(f'Wrong agency id {agency_id} for report with id {report_id}',
                           error_code='UNSUITABLE_AGENCY')
        except FileNotFoundException:
            raise NotFound(f'Can\'t find mds file for report with id {report_id}', error_code='REPORT_FILE_NOT_FOUND')
        except ReportNotReadyException:
            raise NotFound(f'Report with id {report_id} is not ready', error_code='REPORT_NOT_READY')


@dataclass
class GetClientBonusesReportInfo:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.client_bonuses.value,), feature_flag_name='BONUSES_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int, report_id: int) -> ReportInfo:
        try:
            return await self.sd.client_bonuses.get_detailed_report_info(agency_id, report_id)
        except NoSuchReportException:
            raise NotFound(f'Can\'t find report with id {report_id}')
        except UnsuitableAgencyException:
            raise NotFound(f'Wrong agency id {agency_id} for report with id {report_id}',
                           error_code='UNSUITABLE_AGENCY')


@dataclass
class CreateReport:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.client_bonuses.value,), feature_flag_name='BONUSES_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int, report_json: dict) -> ReportInfo:
        report_info = await self.sd.client_bonuses.create_report(
            agency_id=agency_id,
            name=report_json.get('name'),
            period_from=report_json.get('period_from'),
            period_to=report_json.get('period_to'),
            client_type=report_json.get('client_type')
        )

        return report_info


@dataclass
class DeleteReport:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.client_bonuses.value,), feature_flag_name='BONUSES_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int, report_id: int) -> bool:
        try:
            is_deleted = await self.sd.client_bonuses.delete_report(
                agency_id=agency_id,
                report_id=report_id
            )
            return is_deleted
        except NoSuchReportException:
            raise NotFound(f'Can\'t find report with id {report_id}')
        except UnsuitableAgencyException:
            raise NotFound(f'Wrong agency id {agency_id} for report with id {report_id}',
                           error_code='UNSUITABLE_AGENCY')


@dataclass
class ListCashbackPrograms:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.client_bonuses.value,), feature_flag_name='BONUSES_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int) -> list[CashbackProgram]:
        return await self.sd.client_bonuses.list_cashback_programs(agency_id)
