from dataclasses import dataclass
from typing import Optional

from crm.agency_cabinet.certificates.client import CertificateNotFound
from crm.agency_cabinet.certificates.common.structs import (
    AgencyCertificate,
    AgencyCertificateDetails,
    AgencyCertificatesHistoryEntry,
    CertifiedEmployee,
)
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants, _check_grants
from crm.agency_cabinet.grants.common.consts import Permissions

__all__ = [
    "ListAgencyCertificates",
    "ListEmployeesCertificates",
    "FetchAgencyCertificatesHistory",
    "GetAgencyCertificatesDetails",
]


@dataclass
class ListEmployeesCertificates:
    __slots__ = "sd"

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.certification.value,), feature_flag_name='CERTIFICATES_API_NEW_GRANTS')
    async def __call__(
        self,
        yandex_uid: int,
        agency_id: int,
        offset: int,
        limit: int,
        project: Optional[str],
        search_query: Optional[str],
        status: Optional[str],
    ) -> list[CertifiedEmployee]:

        return await self.sd.certificates.list_employees_certificates(
            agency_id=agency_id,
            offset=offset,
            limit=limit,
            project=project,
            search_query=search_query,
            status=status,
        )


@dataclass
class ListAgencyCertificates:
    __slots__ = "sd"

    sd: ServiceDiscovery

    async def __call__(
        self, yandex_uid: int, agency_id: int
    ) -> list[AgencyCertificate]:
        agency_info = await self.sd.agencies.get_agencies_info(agency_ids=[agency_id])

        if not agency_info:
            raise NotFound()
        # TODO: понять почему такой порядок и почему надо уметь возвращать 404 без проверки прав
        await _check_grants(
            yandex_uid,
            agency_id,
            [Permissions.certification.value],
            self.sd,
            'CERTIFICATES_API_NEW_GRANTS'
        )

        return await self.sd.certificates.list_agency_certificates(agency_id=agency_id)


@dataclass
class FetchAgencyCertificatesHistory:
    __slots__ = "sd"

    sd: ServiceDiscovery

    async def __call__(
        self,
        yandex_uid: int,
        agency_id: int,
        project: Optional[str],
        offset: int,
        limit: int,
    ) -> list[AgencyCertificatesHistoryEntry]:
        agency_info = await self.sd.agencies.get_agencies_info(agency_ids=[agency_id])

        if not agency_info:
            raise NotFound()

        # TODO: понять почему такой порядок и почему надо уметь возвращать 404 без проверки прав
        await _check_grants(
            yandex_uid,
            agency_id,
            [Permissions.certification.value],
            self.sd,
            'CERTIFICATES_API_NEW_GRANTS'
        )

        return await self.sd.certificates.fetch_agency_certificates_history(
            agency_id=agency_id,
            project=project,
            offset=offset,
            limit=limit,
        )


@dataclass
class GetAgencyCertificatesDetails:
    __slots__ = "sd"

    sd: ServiceDiscovery

    async def __call__(
        self, yandex_uid: int, agency_id: int
    ) -> AgencyCertificateDetails:
        agency_info = await self.sd.agencies.get_agencies_info(agency_ids=[agency_id])

        if not agency_info:
            raise NotFound("Agency is not found")

        # TODO: понять почему такой порядок и почему надо уметь возвращать 404 без проверки прав
        await _check_grants(
            yandex_uid,
            agency_id,
            [Permissions.certification.value],
            self.sd,
            'CERTIFICATES_API_NEW_GRANTS'
        )

        try:
            return await self.sd.certificates.fetch_agency_certificate_details(
                agency_id=agency_id
            )
        except CertificateNotFound:
            raise NotFound("Certificate is not found")
