from dataclasses import dataclass
from datetime import datetime

from crm.agency_cabinet.documents.common.structs import (
    Act,
)
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants
from crm.agency_cabinet.grants.common.consts import Permissions
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound
from crm.agency_cabinet.documents.client import UnsuitableAgencyException, NoSuchActException, \
    FileNotFoundException

__all__ = ['ListActs', 'GetActUrl']


@dataclass
class ListActs:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.documents.value, ))
    async def __call__(
        self,
        yandex_uid: int,
        agency_id: int,
        limit: int = None,
        offset: int = None,
        contract_id: int = None,
        invoice_id: int = None,
        date_from: datetime = None,
        date_to: datetime = None,
        search_query: str = None,
    ) -> list[Act]:

        return await self.sd.documents.list_acts(
            agency_id=agency_id,
            limit=limit,
            contract_id=contract_id,
            invoice_id=invoice_id,
            offset=offset,
            date_from=date_from,
            date_to=date_to,
            search_query=search_query,

        )


@dataclass
class GetActUrl:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.documents.value, ))
    async def __call__(self, yandex_uid: int, agency_id: int, act_id: int) -> str:
        try:
            url = await self.sd.documents.get_act_url(
                agency_id=agency_id,
                act_id=act_id
            )
            return url
        except NoSuchActException:
            raise NotFound(f'Can\'t find act with id {act_id}')
        except UnsuitableAgencyException:
            raise NotFound(f'Wrong agency id {agency_id} for act with id {act_id}',
                           error_code='UNSUITABLE_AGENCY')
        except FileNotFoundException:
            raise NotFound(f'Can\'t find mds file for act with id {act_id}',
                           error_code='DOCUMENT_FILE_NOT_FOUND')
