from dataclasses import dataclass
from datetime import datetime

from crm.agency_cabinet.documents.common.structs import (
    Agreement,
)
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants
from crm.agency_cabinet.grants.common.consts import Permissions
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound
from crm.agency_cabinet.documents.client import UnsuitableAgencyException, \
    FileNotFoundException, NoSuchAgreementException

__all__ = ['ListAgreements', 'GetAgreementUrl']


@dataclass
class ListAgreements:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.documents.value, ))
    async def __call__(
        self,
        yandex_uid: int,
        agency_id: int,
        limit: int = None,
        offset: int = None,
        contract_id: int = None,
        date_from: datetime = None,
        date_to: datetime = None,
        search_query: str = None,
    ) -> list[Agreement]:

        return await self.sd.documents.list_agreements(
            agency_id=agency_id,
            limit=limit,
            contract_id=contract_id,
            offset=offset,
            date_from=date_from,
            date_to=date_to,
            search_query=search_query,

        )


@dataclass
class GetAgreementUrl:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.documents.value, ))
    async def __call__(self, yandex_uid: int, agency_id: int, agreement_id: int) -> str:
        try:
            url = await self.sd.documents.get_agreement_url(
                agency_id=agency_id,
                agreement_id=agreement_id
            )
            return url
        except NoSuchAgreementException:
            raise NotFound(f'Can\'t find agreement with id {agreement_id}')
        except UnsuitableAgencyException:
            raise NotFound(f'Wrong agency id {agency_id} for agreement with id {agreement_id}',
                           error_code='UNSUITABLE_AGENCY')
        except FileNotFoundException:
            raise NotFound(f'Can\'t find mds file for agreement with id {agreement_id}',
                           error_code='DOCUMENT_FILE_NOT_FOUND')
