from dataclasses import dataclass

from crm.agency_cabinet.documents.common.structs import (
    Contract,
)
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound
from crm.agency_cabinet.documents.client.exceptions import NoSuchContractException, UnsuitableAgencyException, \
    FileNotFoundException
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants
from crm.agency_cabinet.grants.common.consts import Permissions

__all__ = ['ListContracts', 'GetContractInfo', 'GetContractUrl']


@dataclass
class ListContracts:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.documents.value,))
    async def __call__(self, yandex_uid: int, agency_id: int) -> list[Contract]:

        return await self.sd.documents.list_contracts(agency_id=agency_id)


@dataclass
class GetContractInfo:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.documents.value,))
    async def __call__(self, yandex_uid: int, agency_id: int, contract_id: int) -> Contract:
        try:
            return await self.sd.documents.get_contract_info(
                agency_id=agency_id,
                contract_id=contract_id
            )
        except NoSuchContractException:
            raise NotFound(f'Can\'t find contract with id {contract_id}')
        except UnsuitableAgencyException:
            raise NotFound(f'Wrong agency id {agency_id} for contract with id {contract_id}',
                           error_code='UNSUITABLE_AGENCY')


@dataclass
class GetContractUrl:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.documents.value,))
    async def __call__(self, yandex_uid: int, agency_id: int, contract_id: int) -> str:
        try:
            url = await self.sd.documents.get_contract_url(
                agency_id=agency_id,
                contract_id=contract_id
            )
            return url
        except NoSuchContractException:
            raise NotFound(f'Can\'t find contract with id {contract_id}')
        except UnsuitableAgencyException:
            raise NotFound(f'Wrong agency id {agency_id} for contract with id {contract_id}',
                           error_code='UNSUITABLE_AGENCY')
        except FileNotFoundException:
            raise NotFound(f'Can\'t find mds file for contract with id {contract_id}',
                           error_code='DOCUMENT_FILE_NOT_FOUND')
