from dataclasses import dataclass
from datetime import date

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.documents.client import NoSuchInvoiceException, UnsuitableAgencyException, \
    FileNotFoundException
from crm.agency_cabinet.documents.common.structs import Invoice, InvoiceStatus, DetailedInvoice
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants
from crm.agency_cabinet.grants.common.consts import Permissions


__all__ = ['ListInvoices', 'GetInvoiceInfo', 'GetFactureUrl']


@dataclass
class ListInvoices:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.documents.value,))
    async def __call__(self,
                       yandex_uid: int,
                       agency_id: int,
                       contract_id: int = None,
                       date_from: date = None,
                       date_to: date = None,
                       status: InvoiceStatus = None,
                       limit: int = None,
                       offset: int = None,
                       search_query: str = None) -> list[Invoice]:

        return await self.sd.documents.list_invoices(
            agency_id=agency_id,
            contract_id=contract_id,
            date_from=date_from,
            date_to=date_to,
            status=status,
            limit=limit,
            offset=offset,
            search_query=search_query,
        )


@dataclass
class GetInvoiceInfo:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.documents.value,))
    async def __call__(self, yandex_uid: int, agency_id: int, invoice_id: int) -> DetailedInvoice:
        try:
            return await self.sd.documents.get_invoice_info(
                agency_id=agency_id,
                invoice_id=invoice_id
            )

        except NoSuchInvoiceException:
            raise NotFound(f'Can\'t find invoice with id {invoice_id}')

        except UnsuitableAgencyException:
            raise NotFound(f'Wrong agency id {agency_id} for invoice with id {invoice_id}',
                           error_code='UNSUITABLE_AGENCY')


@dataclass
class GetInvoiceUrl:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.documents.value,))
    async def __call__(self, yandex_uid: int, agency_id: int, invoice_id: int) -> str:
        try:
            url = await self.sd.documents.get_invoice_url(
                agency_id=agency_id,
                invoice_id=invoice_id
            )
            return url
        except NoSuchInvoiceException:
            raise NotFound(f'Can\'t find invoice with id {invoice_id}')
        except UnsuitableAgencyException:
            raise NotFound(f'Wrong agency id {agency_id} for invoice with id {invoice_id}',
                           error_code='UNSUITABLE_AGENCY')
        except FileNotFoundException:
            raise NotFound(f'Can\'t find mds file for invoice with id {invoice_id}',
                           error_code='DOCUMENT_FILE_NOT_FOUND')


@dataclass
class GetFactureUrl:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.documents.value,))
    async def __call__(self, yandex_uid: int, agency_id: int, facture_id: int) -> str:
        try:
            url = await self.sd.documents.get_facture_url(
                agency_id=agency_id,
                facture_id=facture_id
            )
            return url
        except NoSuchInvoiceException:
            raise NotFound(f'Can\'t find facture with id {facture_id}')
        except UnsuitableAgencyException:
            raise NotFound(f'Wrong agency id {agency_id} for facture with id {facture_id}',
                           error_code='UNSUITABLE_AGENCY')
        except FileNotFoundException:
            raise NotFound(f'Can\'t find mds file for facture with id {facture_id}',
                           error_code='DOCUMENT_FILE_NOT_FOUND')
