import typing
from decimal import Decimal
from dataclasses import dataclass

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.ord.common import structs
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound
from crm.agency_cabinet.ord.client.exceptions import NoSuchActException, UnsuitableReportException
from crm.agency_cabinet.grants.common.consts import Permissions
from crm.agency_cabinet.common.proto_utils import safe_wrapper_coercion
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants


@dataclass
class AddAct:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(self,
                       yandex_uid: int,
                       agency_id: int,
                       report_id: int,
                       act_eid: str,
                       amount: typing.Optional[float] = None,
                       is_vat: typing.Optional[bool] = None,
                       ) -> structs.Act:
        try:
            return await self.sd.ord.add_act(
                agency_id=agency_id,
                report_id=report_id,
                act_eid=act_eid,
                amount=safe_wrapper_coercion(amount, Decimal),
                is_vat=is_vat,
            )
        except UnsuitableReportException:
            raise NotFound(f'Can\'t find report {report_id}')


@dataclass
class EditAct:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(self,
                       yandex_uid: int,
                       agency_id: int,
                       report_id: int,
                       act_id: int,
                       act_eid: str,
                       amount: typing.Optional[float] = None,
                       is_vat: typing.Optional[bool] = None,
                       ) -> typing.NoReturn:
        try:
            await self.sd.ord.edit_act(
                agency_id=agency_id,
                report_id=report_id,
                act_id=act_id,
                act_eid=act_eid,
                amount=safe_wrapper_coercion(amount, Decimal),
                is_vat=is_vat,
            )
        except NoSuchActException:
            raise NotFound(f'Can\'t find act {act_id} for report {report_id}')
        except UnsuitableReportException:
            raise NotFound(f'Can\'t find report {report_id} for agency {agency_id}')


@dataclass
class GetActs:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(
        self,
        yandex_uid: int,
        agency_id: int,
        report_id: int,
        limit: int = None,
        offset: int = None,
        search_query: str = None,
    ) -> structs.ActList:
        return await self.sd.ord.get_acts(
            agency_id=agency_id,
            report_id=report_id,
            limit=limit,
            offset=offset,
            search_query=search_query,
        )
