import typing
from dataclasses import dataclass

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound, Conflict
from crm.agency_cabinet.ord.common import structs
from crm.agency_cabinet.ord.client.exceptions import NoSuchReportException, UnsuitableAgencyException, OrdException, \
    NoSuchReportRowException, UniqueViolationReportRowException, UnsuitableClientException, UnsuitableReportException
from crm.agency_cabinet.grants.common.consts import Permissions
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants


@dataclass
class ListClientRows:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,))
    async def __call__(self,
                       yandex_uid: int,
                       agency_id: int,
                       report_id: int,
                       client_id: int,
                       limit: int = None,
                       offset: int = None,
                       search_query: str = None,
                       ) -> structs.ClientRowsList:
        return await self.sd.ord.get_client_rows(
            agency_id=agency_id,
            report_id=report_id,
            client_id=client_id,
            limit=limit,
            offset=offset,
            search_query=search_query
        )


@dataclass
class EditClientRow:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,))
    async def __call__(self,
                       yandex_uid: int,
                       agency_id: int,
                       report_id: int,
                       client_id: int,
                       row_id: int,
                       client_contract_id: typing.Optional[int] = None,
                       client_act_id:  typing.Optional[int] = None,
                       ad_distributor_act_id:  typing.Optional[int] = None,
                       campaign_eid: typing.Optional[str] = None,
                       campaign_name: typing.Optional[str] = None,
                       ):
        try:
            await self.sd.ord.edit_client_row(
                agency_id=agency_id,
                report_id=report_id,
                client_id=client_id,
                row_id=row_id,
                client_contract_id=client_contract_id,
                client_act_id=client_act_id,
                ad_distributor_act_id=ad_distributor_act_id,
                campaign_eid=campaign_eid,
                campaign_name=campaign_name,
            )
        except UnsuitableAgencyException:
            raise NotFound(f'Can\'t find row {row_id} for agency {agency_id}')
        except NoSuchReportException:
            raise NotFound(f'Can\'t find row {row_id} for report {report_id}')
        except UnsuitableClientException:
            raise NotFound(f'Can\'t find row {row_id} for client {client_id}')
        except UnsuitableReportException:
            raise NotFound(f'Can\'t find report {report_id} or you are trying to modify Yandex report')
        except NoSuchReportRowException:
            raise NotFound(f'Can\'t find row {row_id}')
        except UniqueViolationReportRowException:
            raise Conflict('There is row with same params')
        except OrdException:
            raise Conflict('Dont do that')
