import typing
from dataclasses import dataclass

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound, Conflict
from crm.agency_cabinet.ord.common import structs
from crm.agency_cabinet.ord.client.exceptions import NoSuchReportException, UnsuitableAgencyException, \
    UnsuitableClientException, UnsuitableReportException, UniqueViolationClientException, \
    ForbiddenByReportSettingsException
from crm.agency_cabinet.grants.common.consts import Permissions
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants


@dataclass
class ListReportClients:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(
        self,
        yandex_uid: int,
        agency_id: int,
        report_id: int,
        is_valid: bool = None,
        limit: int = None,
        offset: int = None,
        search_query: str = None
    ) -> typing.List[structs.ClientInfo]:
        return await self.sd.ord.get_report_clients_info(
            agency_id=agency_id,
            report_id=report_id,
            is_valid=is_valid,
            limit=limit,
            offset=offset,
            search_query=search_query
        )


@dataclass
class GetClientShortInfo:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(
        self,
        yandex_uid: int,
        agency_id: int,
        report_id: int,
        client_id: int,
    ) -> structs.ClientShortInfo:
        try:
            return await self.sd.ord.get_client_short_info(
                agency_id=agency_id,
                report_id=report_id,
                client_id=client_id,
            )
        except UnsuitableClientException:
            raise NotFound(f'Can\'t find client {client_id}')
        except UnsuitableReportException:
            raise NotFound(f'Can\'t find client {client_id} for report {report_id}')
        except UnsuitableAgencyException:
            raise NotFound(f'Can\'t find client {client_id} for agency {agency_id}')


@dataclass
class CreateClient:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(self,
                       yandex_uid: int,
                       agency_id: int,
                       report_id: int,
                       client_id: str,
                       login: typing.Optional[str] = None,
                       name: typing.Optional[str] = None,
                       ):

        try:
            return await self.sd.ord.create_client(
                agency_id=agency_id,
                report_id=report_id,
                client_id=client_id,
                login=login,
                name=name,
            )
        except NoSuchReportException:
            raise NotFound(f'Can\'t find report {report_id}')
        except UnsuitableAgencyException:
            raise NotFound(f'Can\'t find report {report_id} is not found for agency {agency_id}')
        except ForbiddenByReportSettingsException:
            raise Conflict('Can\'t modify report due to its settings')
        except UniqueViolationClientException:
            raise Conflict('Client with these params already exists')
