import uuid
from dataclasses import dataclass

from crm.agency_cabinet.grants.common.consts import Permissions
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound, Conflict
from crm.agency_cabinet.ord.client.exceptions import NoSuchReportException, UnsuitableAgencyException, ImportDataAlreadyRequested
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants


@dataclass
class ImportData:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(self, yandex_uid: int, agency_id: int, s3_resource, report_id: int, filename: str, data: bytes,
                       bucket: str, content_type: str):
        filename = filename.replace('/', '-')
        mds_filename = await self._upload_to_mds(s3_resource, filename, data, bucket, content_type)

        try:
            return await self.sd.ord.import_data(
                agency_id=agency_id,
                report_id=report_id,
                filename=mds_filename,
                bucket=bucket,
                display_name=filename
            )
        except UnsuitableAgencyException:
            raise NotFound(f'Unsuitable agency {agency_id}')
        except NoSuchReportException:
            raise NotFound('Report not found')
        except ImportDataAlreadyRequested:
            raise Conflict('Import was already requested')

    async def _upload_to_mds(self, s3_resource, filename: str, data: bytes, bucket: str, content_type: str):
        mds_filename = f'ord/import/{str(uuid.uuid4())}-{filename}'
        obj = await s3_resource.Object(bucket, mds_filename)
        await obj.put(Body=data, ContentType=content_type, ContentDisposition=f'attachment; filename={filename}')
        return mds_filename
