import typing
from datetime import date, datetime, timezone
from dataclasses import dataclass

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound, Conflict
from crm.agency_cabinet.ord.common.structs import ReportStatuses, ReportInfo, ReportSort
from crm.agency_cabinet.ord.client.exceptions import NoSuchReportException, UnsuitableAgencyException, \
    ReportAlreadySentException, ReportNotReadyException, FileNotFoundException
from crm.agency_cabinet.grants.common.consts import Permissions
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants
from crm.agency_cabinet.ord.common.consts import ReporterType


@dataclass
class ListReports:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(
        self,
        yandex_uid: int,
        agency_id: int,
        search_query: str = None,
        period_from: date = None,
        period_to: date = None,
        status: str = None,
        sort: typing.List[ReportSort] = None,
        limit: int = None,
        offset: int = None,
    ) -> typing.List[ReportInfo]:
        period_from = datetime.combine(period_from, datetime.min.time(), tzinfo=timezone.utc) if period_from else None
        period_to = datetime.combine(period_to, datetime.min.time(), tzinfo=timezone.utc) if period_to else None

        return await self.sd.ord.get_reports_info(
            agency_id=agency_id,
            search_query=search_query,
            period_from=period_from,
            period_to=period_to,
            status=ReportStatuses(status) if status is not None else None,
            sort=sort or [],
            limit=limit,
            offset=offset
        )


@dataclass
class DetailedReportInfo:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(self, yandex_uid: int, agency_id: int, report_id: int) -> ReportInfo:
        try:
            return await self.sd.ord.get_detailed_report_info(
                agency_id=agency_id,
                report_id=report_id
            )
        except NoSuchReportException:
            raise NotFound('Report not found')
        except UnsuitableAgencyException:
            raise NotFound(f'Unsuitable agency {agency_id}')


@dataclass
class SendReport:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(self, yandex_uid: int, agency_id: int, report_id: int):
        try:
            await self.sd.ord.send_report(
                agency_id=agency_id,
                report_id=report_id,
            )
        except UnsuitableAgencyException:
            raise NotFound(f'Unsuitable agency {agency_id}')
        except NoSuchReportException:
            raise NotFound('Report not found')
        except ReportAlreadySentException:
            raise Conflict('Report was already sent')


@dataclass
class ReportExport:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(self, yandex_uid: int, agency_id: int, report_id: int):
        try:
            return await self.sd.ord.report_export(
                agency_id=agency_id,
                report_id=report_id
            )
        except UnsuitableAgencyException:
            raise NotFound(f'Unsuitable agency {agency_id}')
        except NoSuchReportException:
            raise NotFound(f'Can\'t find report with {report_id}')


@dataclass
class ReportExportInfo:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(self, yandex_uid: int, agency_id: int, report_id: int, report_export_id: int):
        try:
            return await self.sd.ord.get_report_export_info(
                agency_id=agency_id,
                report_id=report_id,
                report_export_id=report_export_id
            )
        except NoSuchReportException:
            raise NotFound('Report not found')
        except UnsuitableAgencyException:
            raise NotFound(f'Unsuitable agency {agency_id}')


@dataclass
class ReportUrl:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(self, yandex_uid: int, agency_id: int, report_id: int, report_export_id: int) -> str:
        try:
            return await self.sd.ord.get_report_url(
                agency_id=agency_id,
                report_id=report_id,
                report_export_id=report_export_id
            )
        except NoSuchReportException:
            raise NotFound('Report not found')
        except UnsuitableAgencyException:
            raise NotFound(f'Unsuitable agency {agency_id}')
        except ReportNotReadyException:
            raise NotFound(f'Report {report_id} is in progress')
        except FileNotFoundException:
            raise NotFound('Report not found')


@dataclass
class GetLockStatus:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(self, yandex_uid: int, agency_id: int, report_id: int):
        try:
            return await self.sd.ord.get_lock_status(
                agency_id=agency_id,
                report_id=report_id,
            )
        except UnsuitableAgencyException:
            raise NotFound(f'Unsuitable agency {agency_id}')
        except NoSuchReportException:
            raise NotFound('Report not found')


@dataclass
class DeleteReport:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(self, yandex_uid: int, agency_id: int, report_id: int):
        try:
            await self.sd.ord.delete_report(
                agency_id=agency_id,
                report_id=report_id
            )
        except UnsuitableAgencyException:
            raise NotFound(f'Can\'t find report {report_id} for agency {agency_id}')
        except NoSuchReportException:
            raise NotFound(f'Can\'t find report {report_id}')


@dataclass
class CreateReport:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.ord.value,), force_original_parameter=True)
    async def __call__(self,
                       yandex_uid: int,
                       agency_id: int,
                       period_from: date,
                       reporter_type: str,
                       ):
        period_from = datetime.combine(period_from, datetime.min.time(), tzinfo=timezone.utc)
        return await self.sd.ord.create_report(
            agency_id=agency_id,
            period_from=period_from,
            reporter_type=ReporterType(reporter_type),
        )
