import typing

from dataclasses import dataclass
from crm.agency_cabinet.common.consts import CalculatorServiceType
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.grants.common.consts import Permissions
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound, InternalServerError
from crm.agency_cabinet.rewards.client.exceptions import BunkerError, NoSuchCalculatorDataException, NoSuchContractException, BunkerNotFoundException
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants
from crm.agency_cabinet.rewards.common import structs


@dataclass
class CalculatorData:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.rewards.value,), feature_flag_name='CALCULATOR_API_NEW_GRANTS')
    async def __call__(
        self,
        yandex_uid: int,
        agency_id: int,
        contract_id: int,
        service: typing.Union[str, CalculatorServiceType],
        version: str
    ) -> str:
        try:
            data = await self.sd.rewards.get_calculator_data(
                agency_id=agency_id,
                contract_id=contract_id,
                service=service,
                version=version
            )
            return data.result

        except NoSuchCalculatorDataException:
            raise NotFound()


@dataclass
class CalculatorMeta:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.rewards.value,), feature_flag_name='CALCULATOR_API_NEW_GRANTS')
    async def __call__(
        self,
        yandex_uid: int,
        agency_id: int,
        contract_id: int,
        service: typing.Union[str, CalculatorServiceType],
        version: str
    ) -> str:

        try:
            meta: structs.GetCalculatorMetaResponse = await self.sd.rewards.get_calculator_meta(
                agency_id=agency_id, contract_id=contract_id, service=service, version=version
            )
            return meta.result
        except BunkerError as e:
            raise InternalServerError(str(e))
        except NoSuchContractException:
            raise NotFound(f'Contract {contract_id} for agency {agency_id} is not found')
        except BunkerNotFoundException:
            raise NotFound('Calculator meta is not found')


@dataclass
class ListAvailableCalculators:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.rewards.value,), feature_flag_name='CALCULATOR_API_NEW_GRANTS')
    async def __call__(
        self,
        yandex_uid: int,
        agency_id: int,
        contract_id: int = None,
    ) -> structs.ListAvailableCalculatorsResponse:
        return await self.sd.rewards.list_available_calculators(
            agency_id=agency_id,
            contract_id=contract_id
        )
