from dataclasses import dataclass

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound
from crm.agency_cabinet.rewards.client import UnsuitableAgency, FileNotFoundException, NoSuchDocumentException
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants
from crm.agency_cabinet.grants.common.consts import Permissions


@dataclass
class GetDocumentUrl:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.rewards.value,), feature_flag_name='DOCUMENTS_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int, document_id: int) -> str:
        try:
            url = await self.sd.rewards.get_document_url(agency_id, document_id)
            return url
        except NoSuchDocumentException:
            raise NotFound(f'Can\'t find document with id {document_id}')
        except UnsuitableAgency:
            raise NotFound(f'Wrong agency id {agency_id} for document with id {document_id}', error_code='UNSUITABLE_AGENCY')
        except FileNotFoundException:
            raise NotFound(f'Can\'t find file for document with id {document_id}', error_code='DOCUMENT_FILE_NOT_FOUND')
