from dataclasses import dataclass
from datetime import datetime, date

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound, Unprocessable
from crm.agency_cabinet.rewards.common import structs as rewards_structs
from crm.agency_cabinet.rewards.client.exceptions import NoSuchReportException, UnsuitableAgency, FileNotFoundException, \
    ReportNotReadyException, NoSuchContractException, UnknownClientsException, UnsupportedReportParametersException
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants
from crm.agency_cabinet.grants.common.consts import Permissions


@dataclass
class ListReports:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.rewards.value,), feature_flag_name='REPORTS_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int, contract_id: int = None,
                       type: str = None, service: str = None) -> list[rewards_structs.ReportInfo]:
        reports_info = await self.sd.rewards.get_reports_info(agency_id,
                                                              filter_contract=contract_id,
                                                              filter_type=type,
                                                              filter_service=service)
        return reports_info


@dataclass
class DetailedReportInfo:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.rewards.value,), feature_flag_name='REPORTS_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int, report_id: int) -> rewards_structs.DetailedReportInfo:
        try:
            report_info = await self.sd.rewards.get_detailed_report_info(agency_id, report_id)
        except NoSuchReportException:
            raise NotFound(f'Can\'t find report with id {report_id}')
        except UnsuitableAgency:
            raise NotFound(f'Wrong agency id {agency_id} for report with id {report_id}')

        return report_info


@dataclass
class CreateReport:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.rewards.value,), feature_flag_name='REPORTS_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int, report_json: dict) -> rewards_structs.DetailedReportInfo:

        date_from: date = report_json.get('period_from')
        date_to: date = report_json.get('period_to')

        period_from = datetime.combine(date_from, datetime.min.time()) if date_from else None
        period_to = datetime.combine(date_to, datetime.max.time()) if date_to else None
        try:
            report_info = await self.sd.rewards.create_report(
                agency_id,
                report_json['contract_id'],
                report_json['name'],
                report_json['type'],
                report_json['service'],
                period_from,
                period_to,
                report_json.get('clients')
            )
        except UnsuitableAgency:
            raise NotFound(f'Wrong agency id {agency_id} for contract with id {report_json["contract_id"]}',
                           error_code='UNSUITABLE_AGENCY')
        except NoSuchContractException:
            raise NotFound(f'Can\'t find contract with id {report_json["contract_id"]}',
                           error_code='UNKNOWN_CONTRACT')
        except UnknownClientsException:
            raise NotFound(f'Can\'t find clients: {report_json.get("clients")}',
                           error_code='UNKNOWN_CLIENTS')
        except UnsupportedReportParametersException as ex:
            raise Unprocessable(f'It is impossible to build report: {ex}',
                                error_code='IMPOSSIBLE_REPORT')
        return report_info


@dataclass
class DeleteReport:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.rewards.value,), feature_flag_name='REPORTS_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int, report_id: int) -> bool:
        try:
            is_deleted = await self.sd.rewards.delete_report(agency_id, report_id)
            return is_deleted
        except NoSuchReportException:
            raise NotFound(f'Can\'t find report with id {report_id}')
        except UnsuitableAgency:
            raise NotFound(f'Wrong agency id {agency_id} for report with id {report_id}')


@dataclass
class GetReportUrl:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.rewards.value,), feature_flag_name='REPORTS_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int, report_id: int) -> str:
        try:
            url = await self.sd.rewards.get_report_url(agency_id, report_id)
            return url
        except NoSuchReportException:
            raise NotFound(f'Can\'t find report with id {report_id}')
        except UnsuitableAgency:
            raise NotFound(f'Wrong agency id {agency_id} for report with id {report_id}', error_code='UNSUITABLE_AGENCY')
        except FileNotFoundException:
            raise NotFound(f'Can\'t find mds file for report with id {report_id}', error_code='REPORT_FILE_NOT_FOUND')
        except ReportNotReadyException:
            raise NotFound(f'Report with id {report_id} is not ready', error_code='REPORT_NOT_READY')
