from dataclasses import dataclass
from datetime import date, datetime

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.exceptions import NotFound, Unprocessable
from crm.agency_cabinet.rewards.common import structs as rewards_structs

from crm.agency_cabinet.rewards.client import NoSuchRewardException, UnsuitableAgency, UnsupportedParametersException
from crm.agency_cabinet.gateway.server.src.procedures.common import check_grants
from crm.agency_cabinet.grants.common.consts import Permissions


@dataclass
class ListRewards:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.rewards.value,), feature_flag_name='REWARDS_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int, contract_id: int = None,
                       date_from: date = None, date_to: date = None,
                       type: str = None, is_paid: bool = None) -> list[rewards_structs.RewardInfo]:
        filter_from = datetime.combine(date_from, datetime.min.time()) if date_from else None
        filter_to = datetime.combine(date_to, datetime.min.time()) if date_to else None
        rewards_info = await self.sd.rewards.get_rewards_info(agency_id,
                                                              filter_from=filter_from,
                                                              filter_to=filter_to,  # TODO: rename filter_* -> *
                                                              filter_contract=contract_id,
                                                              filter_type=type,
                                                              filter_is_paid=is_paid)
        return rewards_info


@dataclass
class DetailedRewardInfo:
    __slots__ = "sd"

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.rewards.value,), feature_flag_name='REWARDS_API_NEW_GRANTS')
    async def __call__(self, yandex_uid: int, agency_id: int, reward_id: int) -> rewards_structs.DetailedRewardInfo:
        try:
            reward_info = await self.sd.rewards.get_detailed_reward_info(agency_id, reward_id)
        except NoSuchRewardException:
            raise NotFound(f'Reward with id equal to {reward_id} is not found')
        except UnsuitableAgency:
            raise NotFound(f'There is no reward with the id equal to {reward_id} belonging to the agency with the id '
                           f'equal to {agency_id}')

        return reward_info


@dataclass
class GetDashboard:
    __slots__ = "sd"

    sd: ServiceDiscovery

    @check_grants(permissions_list=(Permissions.rewards.value,), feature_flag_name='REWARDS_API_NEW_GRANTS')
    async def __call__(
            self, yandex_uid: int, agency_id: int, year: int, contract_id: int = None, service: str = None
    ) -> list[rewards_structs.DashboardItem]:
        try:
            dashboard = await self.sd.rewards.get_dashboard(agency_id, year, contract_id, service)
        except UnsupportedParametersException:
            raise Unprocessable('Unsupported parameters', error_code='UNSUPPORTED PARAMETERS')
        return dashboard
