import typing
from dataclasses import dataclass

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway.server.src.exceptions import BadRequest, NotFound, AccessDenied
from crm.agency_cabinet.grants.common import structs as grants_structs
from crm.agency_cabinet.grants.client import (
    NoSuchRoleException, NoSuchUserException,
    NoSuchPermissionException, UnsuitablePermissionException, NotEditablePermissionModificationException,
    NotHavePermission
)
from crm.agency_cabinet.gateway.server.src.procedures.common import tmp_decorator_to_switch_partner_id


@dataclass
class ListPermissions:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @tmp_decorator_to_switch_partner_id
    async def __call__(
        self, yandex_uid: int, agency_id: int = None, search_query: str = None, partner_id: int = None
    ) -> list[grants_structs.UserPermissions]:
        try:
            return await self.sd.grants.get_users_permissions(
                agency_id=agency_id,
                yandex_uid=yandex_uid,
                partner_id=partner_id,
                query=search_query)
        except NotHavePermission as e:
            raise AccessDenied(e.message)


@dataclass
class EditUser:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @staticmethod
    def roles_to_struct(roles: list):
        return [
            grants_structs.InputRole(
                role_id=role['role_id'],
                permissions=[
                    grants_structs.Permission(name=permission['name']) for permission in role['permissions']
                ]
            ) for role in roles
        ]

    @tmp_decorator_to_switch_partner_id
    async def __call__(self, yandex_uid: int, agency_id: int, user_id: int, roles: list, partner_id: int = None):

        roles = self.roles_to_struct(roles)

        try:
            is_edited = await self.sd.grants.edit_user(
                agency_id=agency_id,
                yandex_uid=yandex_uid,
                user_id=user_id,
                roles=roles,
                partner_id=partner_id
            )
            return is_edited
        except NoSuchUserException as e:
            raise NotFound(e.message)
        except NoSuchRoleException as e:
            raise NotFound(e.message)
        except NoSuchPermissionException as e:
            raise NotFound(e.message)
        except UnsuitablePermissionException as e:
            raise BadRequest(e.message)
        except NotEditablePermissionModificationException as e:
            raise BadRequest(e.message)
        except NotHavePermission as e:
            raise AccessDenied(e.message)


@dataclass
class ListSuggestUsers:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @tmp_decorator_to_switch_partner_id
    async def __call__(self, yandex_uid: int, agency_id: int, search_query: str = None, partner_id: int = None):
        try:
            return await self.sd.grants.get_suggested_users(
                agency_id=agency_id,
                yandex_uid=yandex_uid,
                partner_id=partner_id,
                query=search_query)
        except NotHavePermission as e:
            raise AccessDenied(e.message)


@dataclass
class ListRoles:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @tmp_decorator_to_switch_partner_id
    async def __call__(self, yandex_uid: int, agency_id: int = None, partner_id: int = None):
        try:
            return await self.sd.grants.list_roles(
                partner_id=partner_id,
                agency_id=agency_id, yandex_uid=yandex_uid)
        except NotHavePermission as e:
            raise AccessDenied(e.message)


@dataclass
class ListUserRoles:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    @tmp_decorator_to_switch_partner_id
    async def __call__(self, yandex_uid: int, agency_id: int = None, partner_id: int = None):
        try:
            return await self.sd.grants.list_user_roles(agency_id=agency_id, yandex_uid=yandex_uid, partner_id=partner_id)
        except NoSuchUserException as e:
            raise NotFound(e.message)
        except NotHavePermission as e:
            raise AccessDenied(e.message)


@dataclass
class ListPartners:
    __slots__ = 'sd'

    sd: ServiceDiscovery

    async def __call__(self, yandex_uid: int) -> typing.List[grants_structs.Partner]:
        try:
            data = await self.sd.grants.list_available_partners(yandex_uid=yandex_uid)
            return data.partners
        except NoSuchUserException:
            return []
