import logging

from typing import Any
from dataclasses import dataclass, field
from marshmallow import Schema, fields, post_load

LOGGER = logging.getLogger('middlewares.error')


@dataclass
class ErrorMessage:
    text: str
    params: dict[str, Any] = field(default_factory=dict)

    class _SafeDict(dict):
        def __missing__(self, key):
            LOGGER.warning('Value for key {key} is missing in params'.format(key=key))
            return 'MISSING_VALUE'

    def process_message(self):
        try:
            return self.text.format(**self.params)
        except KeyError:
            try:
                return self.text.format_map(self._SafeDict(self.params))
            except ValueError:
                return self.text


@dataclass
class ErrorDetails:
    http_code: int
    error_code: str
    messages: list[ErrorMessage]


class ErrorMessageSchema(Schema):
    text = fields.String(required=True)
    params = fields.Dict(required=True)

    @post_load
    def create_error_message(self, data, **kwargs):
        return ErrorMessage(**data)


class ErrorDetailsSchema(Schema):
    http_code = fields.Integer(required=True)
    error_code = fields.String(required=True)
    messages = fields.List(fields.Nested(ErrorMessageSchema))

    @post_load
    def create_error_details(self, data, **kwargs):
        return ErrorDetails(**data)
