import logging.config
import functools
import typing
from aiohttp import web
from crm.agency_cabinet.common.monitoring.utils.registry import setup_monitoring
from crm.agency_cabinet.common.monitoring.utils.server import setup_metrics_registry
from crm.agency_cabinet.common.aiohttp_logging import RequestIdContextAccessLogger
from crm.agency_cabinet.common.server.common.logging_config import get_logging_config
from crm.agency_cabinet.common.blackbox import BlackboxClient
from crm.agency_cabinet.common.server.web.apispec import setup_aiohttp_apispec, setup_aiohttp_subapp_apispec
from crm.agency_cabinet.common.server.web.middlewares import (
    RequestIdMiddleware,
    RateMonitoringMiddleware
)
from crm.agency_cabinet.gateway_external.server.src.middlewares import ValidationMiddleware, \
    ErrorMiddleware, OAuthAppMiddleware
from crm.agency_cabinet.gateway_external.server.src.middlewares.error import process_validation_error
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery

from .config import GatewayExternalConfig
from .handlers import (
    InfraCollection,
    OrdReportsCollection,
)


def run_server():
    cfg = GatewayExternalConfig.from_environ()
    logging.config.dictConfig(get_logging_config(cfg.verbose))

    async def _setup(c):
        sd = ServiceDiscovery(c.amqp_url)
        await sd.connect()
        blackbox_client = BlackboxClient(c.tvm2, c.blackbox)
        return await setup_app(c, sd, blackbox_client)

    web.run_app(_setup(cfg), port=cfg.port, access_log_class=RequestIdContextAccessLogger)


async def setup_app(cfg: GatewayExternalConfig,
                    service_discovery: ServiceDiscovery,
                    blackbox_client: BlackboxClient) -> web.Application:
    app = web.Application(
        middlewares=[
            RateMonitoringMiddleware(),
            RequestIdMiddleware(),
            ErrorMiddleware(),
        ]
    )

    setup_metrics_registry_bounded = functools.partial(setup_metrics_registry, metric_registry=setup_monitoring())
    app.on_startup.append(setup_metrics_registry_bounded)

    infra_handlers = InfraCollection(service_discovery)
    app.add_routes(
        [
            web.get('/ping', infra_handlers.ping, allow_head=False),
            web.get('/health_check', infra_handlers.health_check, allow_head=False),
        ]
    )

    api = await setup_api_app(cfg,
                              service_discovery,
                              blackbox_client,
                              [setup_metrics_registry_bounded])
    app.add_subapp('/api/', api)
    setup_aiohttp_apispec(
        app=app,
        title="Agency Cabinet External API",
        version="v1",
        url="/docs/external/swagger.json",  # дока по всем роутам app и его subapp
        swagger_path="/docs/external/",
        error_callback=process_validation_error,
    )
    setup_aiohttp_subapp_apispec(
        app,
        api
    )

    return app


async def setup_api_app(
    cfg: GatewayExternalConfig,
    service_discovery: ServiceDiscovery,
    blackbox_client: BlackboxClient,
    on_startup: list[typing.Callable]
) -> web.Application:

    api = web.Application(
        middlewares=[
            RateMonitoringMiddleware(),
            RequestIdMiddleware(),
            ErrorMiddleware(),
            ValidationMiddleware(),
            OAuthAppMiddleware(blackbox_client=blackbox_client, development_mode=cfg.development_mode),
        ]
    )
    for hook in on_startup:
        api.on_startup.append(hook)

    report_handlers = OrdReportsCollection(service_discovery)

    api.add_routes(
        [
            web.get(
                r'/v1/reports', report_handlers.list_reports, allow_head=False
            ),
        ]
    )

    return api
