import logging

from aiohttp import web
from aiohttp_apispec import docs

from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.gateway_external.server.src.exceptions import ProcedureException
from crm.agency_cabinet.gateway_external.server.src.handlers.common import development_headers_docs, json_response
from crm.agency_cabinet.gateway_external.server.src.procedures import infra_procedures
from crm.agency_cabinet.common.monitoring.utils.server import BaseMetricsHandler

LOGGER = logging.getLogger('handlers.external.infra')


class InfraCollection:
    def __init__(self, sd: ServiceDiscovery):
        self._healthcheck = infra_procedures.HealthCheck(sd)
        self._metrics = BaseMetricsHandler()

    @docs(
        tags=['infra'],
        summary='Пинг',
    )
    @development_headers_docs
    async def ping(self, request: web.Request) -> web.Response:
        return json_response({'status': 'ok'})

    @docs(
        tags=['infra'],
        summary='Health check',
    )
    @development_headers_docs
    async def health_check(self, request: web.Request) -> web.Response:
        result = self._healthcheck()
        return json_response({'status': result})

    @docs(
        tags=['infra'],
        summary='Метрики для Solomon',
    )
    @development_headers_docs
    async def metrics(self, request: web.Request) -> web.Response:
        # TODO: allow only for localhost?
        return await self._metrics(request)

    @docs(
        tags=['infra'],
        summary='Кидает неопределенное исключение при вызове',
    )
    @development_headers_docs
    async def raise_500(self, request: web.Request):
        try:
            raise ValueError("500 error")
        except Exception as ex:
            LOGGER.exception('TEST HANDLER EXCEPTION: %s', ex)
            raise ex

    @docs(
        tags=['infra'],
        summary='Кидает процедурное исключение при вызове',
    )
    @development_headers_docs
    async def raise_procedure_ex(self, request: web.Request):
        try:
            raise ProcedureException('Hello world', http_code=200, error_code='RAISED_EXCEPTION')
        except Exception as ex:
            LOGGER.exception('TEST HANDLER EXCEPTION: %s', ex)
            raise ex
