import typing
import logging
from collections.abc import Iterable
from aiohttp.web import HTTPException, middleware, Request, Response
from marshmallow import ValidationError, Schema

from crm.agency_cabinet.gateway_external.server.src.exceptions import InternalServerError, ProcedureException, Unprocessable, MAP_CODE_TO_EXCEPTION
from crm.agency_cabinet.gateway_external.server.src.handlers.common import error_response
from crm.agency_cabinet.common.utils.flatten_dict import flatten_dict

LOGGER = logging.getLogger('server.error')


def build_unprocessable_exception(e: ValidationError):
    validation_messages = e.normalized_messages()
    messages = []
    for key, value in flatten_dict(validation_messages).items():
        if isinstance(value, Iterable):
            value = ';'.join(value)
        messages.append(f"{key}: {value}")
    return Unprocessable(*messages)


def process_validation_error(
    e: ValidationError,
    req: Request,
    schema: Schema,
    error_status_code: typing.Optional[int] = None,
    error_headers: typing.Optional[typing.Mapping[str, str]] = None,
) -> typing.NoReturn:
    raise build_unprocessable_exception(e)


@middleware
class ErrorMiddleware:

    async def __call__(self, request: Request, handler: typing.Callable[[Request], typing.Awaitable[Response]]) -> Response:

        try:
            return await handler(request)
        except ProcedureException as e:
            return error_response(e)
        except ValidationError as e:
            return error_response(build_unprocessable_exception(e))
        except HTTPException as e:
            if e.content_type == 'text/plain' and (exception_cls := MAP_CODE_TO_EXCEPTION.get(e.status_code)):
                return error_response(exception_cls(e.text))
            raise e
        except Exception as e:
            LOGGER.exception('Something went wrong during request processing: %s', e)
            return error_response(
                InternalServerError(str(e))
            )
