import typing
import inspect
import functools
from crm.agency_cabinet.gateway_external.server.src.exceptions import AccessDenied
from crm.agency_cabinet.grants.common.structs import CheckPermissionsResponse
from crm.agency_cabinet.grants.client import (
    NoSuchOAuthTokenException, InactiveOAuthToken, NoSuchPermissionException
)


async def _check_oauth_permissions(app_client_id, permissions_list, sd) -> typing.Optional[CheckPermissionsResponse]:
    try:
        result = await sd.grants.check_oauth_permissions(
            app_client_id=app_client_id,
            permissions=permissions_list,
        )
        if not result.is_have_permissions:
            raise AccessDenied()
        return result
    except NoSuchOAuthTokenException as e:
        raise AccessDenied(e.message)
    except InactiveOAuthToken as e:
        raise AccessDenied(e.message)
    except NoSuchPermissionException as e:
        raise AccessDenied(e.message)


def check_oauth_permissions(permissions_list: typing.Union[typing.List[str], typing.Tuple[str, ...]] = ()):
    def wrapper(func):
        args_name = [name for name in inspect.signature(func).parameters.keys()]
        assert len(args_name) > 2
        assert args_name[0] == 'self'
        assert args_name[1] == 'app_client_id'

        @functools.wraps(func)
        async def decorator(self, app_client_id, *args, **kwargs):
            res = await _check_oauth_permissions(app_client_id, permissions_list, self.sd)
            return await func(self, app_client_id, partner_id=res.partner.partner_id, *args, **kwargs)

        return decorator

    return wrapper
