import typing
import logging.config
import functools
from aiohttp import web

from crm.agency_cabinet.common.server.common.logging_config import get_logging_config
from crm.agency_cabinet.common.server.common.tvm import get_tvm_client, TvmClient
from crm.agency_cabinet.common.server.web.middlewares import TvmServiceMiddleware, RateMonitoringMiddleware, RequestIdMiddleware
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.common.monitoring.utils.server import setup_metrics_registry
from crm.agency_cabinet.common.monitoring.utils.registry import setup_monitoring
from crm.agency_cabinet.common.aiohttp_logging import RequestIdContextAccessLogger


from .config import GatewayInternalConfig
from crm.agency_cabinet.gateway_internal.server.src import handlers


def run_server():
    cfg = GatewayInternalConfig.from_environ()
    logging.config.dictConfig(get_logging_config(cfg.verbose))

    async def _setup(c):
        sd = ServiceDiscovery(c.amqp_url)
        await sd.connect()
        return await setup_app(c, sd, get_tvm_client(c.tvm2))

    web.run_app(_setup(cfg), port=cfg.port, access_log_class=RequestIdContextAccessLogger)


async def setup_app(cfg: GatewayInternalConfig, service_discovery: ServiceDiscovery, tvm_client: TvmClient) -> web.Application:
    app = web.Application()

    infra_handlers = handlers.InfraCollection()
    app.add_routes(
        [
            web.get('/ping/', infra_handlers.ping, allow_head=False)
        ]
    )
    setup_metrics_registry_bounded = functools.partial(setup_metrics_registry, metric_registry=setup_monitoring())
    app.on_startup.append(setup_metrics_registry_bounded)

    app.add_subapp('/idm/', await setup_idm_app(cfg, service_discovery, tvm_client, [setup_metrics_registry_bounded]))
    return app


async def setup_idm_app(cfg: GatewayInternalConfig, service_discovery: ServiceDiscovery, tvm_client: TvmClient, on_startup: list[typing.Callable]) -> web.Application:
    idm_api = web.Application(
        middlewares=[
            RateMonitoringMiddleware(),
            RequestIdMiddleware(),
            TvmServiceMiddleware(tvm_client=tvm_client, allowed_clients=[cfg.idm_client_id])
        ]
    )

    for hook in on_startup:
        idm_api.on_startup.append(hook)

    idm_handlers = handlers.IdmCollection(service_discovery)

    idm_api.add_routes(
        [
            web.get("/info/", idm_handlers.get_info, allow_head=False),
            web.get("/get-all-roles/", idm_handlers.get_all_roles, allow_head=False),
            web.post("/add-role/", idm_handlers.add_role),
            web.post("/remove-role/", idm_handlers.remove_role),
        ]
    )

    return idm_api
