import typing
from crm.agency_cabinet.common.client import BaseClient
from crm.agency_cabinet.grants.proto import common_pb2, request_pb2, access_level_pb2, role_pb2, grants_pb2, partners_pb2
from crm.agency_cabinet.grants.common import QUEUE
from crm.agency_cabinet.grants.common.consts.partner import PartnerType
from crm.agency_cabinet.grants.common import structs


class GrantsClientException(Exception):
    def __init__(self, message=None):
        self.message = message


class ProtocolError(GrantsClientException):
    pass


class ConflictRoleException(GrantsClientException):
    pass


class NoSuchRoleException(GrantsClientException):
    pass


class NoSuchUserException(GrantsClientException):
    pass


class NoSuchPermissionException(GrantsClientException):
    pass


class UnsuitablePermissionException(GrantsClientException):
    pass


class NotEditablePermissionModificationException(GrantsClientException):
    pass


class NotHavePermission(GrantsClientException):
    pass


class PartnerNotFound(GrantsClientException):
    pass


class NoSuchOAuthTokenException(GrantsClientException):
    pass


class InactiveOAuthToken(GrantsClientException):
    pass


class GrantsClient(BaseClient):
    queue = QUEUE

    async def ping(self) -> str:
        _, data = await self._send_message(
            request_pb2.RpcRequest(ping=common_pb2.Empty()),
            common_pb2.PingOutput
        )
        return str(data)

    async def check_access_level(self, yandex_uid: int, agency_id: int) -> structs.AccessLevel:
        request = request_pb2.RpcRequest(
            check_access_level=access_level_pb2.CheckAccessLevel(yandex_uid=yandex_uid, agency_id=agency_id)
        )

        result, data = await self._send_message(
            request,
            access_level_pb2.CheckAccessLevelOutput
        )

        if result != 'access_level':
            raise ProtocolError('Unexpected response')

        return structs.CheckAccessLevelResponse.from_proto(data).level

    async def get_access_scope(self, yandex_uid: int) -> tuple[structs.AccessScopeType, list[int]]:
        request = request_pb2.RpcRequest(
            get_access_scope=access_level_pb2.GetAccessScope(yandex_uid=yandex_uid)
        )

        result, data = await self._send_message(
            request,
            access_level_pb2.GetAccessScopeOutput
        )

        if result != 'result':
            raise ProtocolError('Unexpected response')

        response = structs.GetAccessScopeResponse.from_proto(data)
        return response.scope_type, response.agencies

    async def add_role(self, agency_id: int, email: str, real_ip: str) -> None:
        request = request_pb2.RpcRequest(
            add_role=role_pb2.Role(
                agency_id=agency_id,
                email=email,
                real_ip=real_ip,
            )
        )

        result, data = await self._send_message(
            request,
            role_pb2.AddRoleOutput
        )

        if result == 'success':
            return

        if result == 'conflicting_role_exists':
            raise ConflictRoleException()

        raise ProtocolError('Unexpected response')

    async def remove_role(self, agency_id: int, email: str) -> None:
        request = request_pb2.RpcRequest(
            remove_role=role_pb2.Role(
                agency_id=agency_id,
                email=email,
            )
        )
        # TODO: remove
        result, data = await self._send_message(
            request,
            role_pb2.RemoveRoleOutput
        )

        if result == 'success':
            return

        if result == 'no_such_role':
            raise NoSuchRoleException()

        raise ProtocolError('Unexpected response')

    async def get_all_internal_roles(self) -> list[structs.InternalRole]:
        request = request_pb2.RpcRequest(
            get_all_internal_roles=common_pb2.Empty()
        )

        result, data = await self._send_message(
            request,
            role_pb2.GetAllInternalRolesOutput
        )

        if result == 'result':
            return structs.GetAllInternalRolesResponse.from_proto(data).roles

        raise ProtocolError('Unexpected response')

    async def add_internal_role(self, staff_login: str, email: str, real_ip: str) -> None:
        request = request_pb2.RpcRequest(
            add_internal_role=role_pb2.InternalRole(
                staff_login=staff_login,
                email=email,
                real_ip=real_ip,
            )
        )

        result, data = await self._send_message(
            request,
            role_pb2.AddRoleOutput
        )

        if result == 'success':
            return

        if result == 'conflicting_role_exists':
            raise ConflictRoleException()

        raise ProtocolError('Unexpected response')

    async def remove_internal_role(self, staff_login: str, email: str) -> None:
        request = request_pb2.RpcRequest(
            remove_internal_role=role_pb2.InternalRole(
                staff_login=staff_login,
                email=email,
            )
        )

        result, data = await self._send_message(
            request,
            role_pb2.RemoveRoleOutput
        )

        if result == 'success':
            return

        if result == 'no_such_role':
            raise NoSuchRoleException()

        raise ProtocolError('Unexpected response')

    async def get_users_permissions(
        self,
        yandex_uid: int,
        partner_id: int = None,
        agency_id: int = None,
        query: str = None
    ) -> typing.List[structs.UserPermissions]:
        if partner_id is None and agency_id is None:
            raise ProtocolError('Partner/agency id not set')
        if partner_id is not None:  # prefer partner_id
            agency_id = None

        request = request_pb2.RpcRequest(
            get_users_permissions=structs.GetUsersPermissionsRequest(
                current_yandex_uid=yandex_uid,
                query=query,
                partner_id=partner_id,
                agency_id=agency_id
            ).to_proto()
        )

        result, data = await self._send_message(
            request,
            grants_pb2.GetUsersPermissionsOutput
        )

        if result == 'result':
            return structs.GetUsersPermissionsResponse.from_proto(data).users
        elif result == 'not_have_permission':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NotHavePermission(error_message)

        raise ProtocolError('Unexpected response')

    async def edit_user(
        self,
        yandex_uid: int,
        user_id: int,
        roles: list[structs.InputRole],
        partner_id: int = None,
        agency_id: int = None
    ):
        if partner_id is None and agency_id is None:
            raise ProtocolError('Partner/agency id not set')

        if partner_id is not None:  # prefer partner_id
            agency_id = None

        request = request_pb2.RpcRequest(
            edit_user=structs.EditUserRequest(
                partner_id=partner_id,
                agency_id=agency_id,
                yandex_uid=yandex_uid,
                user_id=user_id,
                roles=roles
            ).to_proto()
        )

        result, data = await self._send_message(
            request,
            grants_pb2.EditUserOutput
        )

        if result == 'result':
            return structs.EditUserResponse.from_proto(data).is_edited
        elif result == 'no_such_user':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NoSuchUserException(error_message)
        elif result == 'no_such_role':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NoSuchUserException(error_message)
        elif result == 'no_such_permission':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NoSuchPermissionException(error_message)
        elif result == 'unsuitable_permission':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise UnsuitablePermissionException(error_message)
        elif result == 'not_editable_permission':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NotEditablePermissionModificationException(error_message)
        elif result == 'not_have_permission':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NotHavePermission(error_message)

        raise ProtocolError('Unexpected response')

    async def get_suggested_users(
        self,
        yandex_uid: int,
        query: str = None,
        partner_id: int = None,
        agency_id: int = None
    ) -> typing.List[structs.User]:
        if partner_id is None and agency_id is None:
            raise ProtocolError('Partner/agency id not set')

        if partner_id is not None:  # prefer partner_id
            agency_id = None

        request = request_pb2.RpcRequest(
            get_suggested_users=structs.GetSuggestedUsersRequest(
                partner_id=partner_id,
                current_yandex_uid=yandex_uid,
                query=query,
                agency_id=agency_id
            ).to_proto()
        )

        result, data = await self._send_message(
            request,
            grants_pb2.GetSuggestedUsersOutput
        )

        if result == 'result':
            return structs.GetSuggestedUsersResponse.from_proto(data).users
        elif result == 'not_have_permission':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NotHavePermission(error_message)

        raise ProtocolError('Unexpected response')

    async def list_roles(self, yandex_uid: int, partner_id: int = None, agency_id: int = None) -> typing.List[structs.RolePermissions]:
        if partner_id is None and agency_id is None:
            raise ProtocolError('Partner/agency id not set')

        if partner_id is not None:  # prefer partner_id
            agency_id = None

        request = request_pb2.RpcRequest(
            list_roles=structs.ListRolesRequest(
                partner_id=partner_id,
                yandex_uid=yandex_uid,
                agency_id=agency_id
            ).to_proto()
        )

        result, data = await self._send_message(
            request,
            role_pb2.ListRolesOutput
        )

        if result == 'result':
            return structs.ListRolesResponse.from_proto(data).roles
        elif result == 'not_have_permission':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NotHavePermission(error_message)

        raise ProtocolError('Unexpected response')

    async def list_user_roles(self, yandex_uid: int, partner_id: int = None, agency_id: int = None) -> typing.List[structs.RolePermissions]:
        if partner_id is None and agency_id is None:
            raise ProtocolError('Partner/agency id not set')

        if partner_id is not None:  # prefer partner_id
            agency_id = None

        request = request_pb2.RpcRequest(
            list_user_roles=structs.ListUserRolesRequest(
                partner_id=partner_id,
                yandex_uid=yandex_uid,
                agency_id=agency_id
            ).to_proto()
        )

        result, data = await self._send_message(
            request,
            role_pb2.ListUserRolesOutput
        )

        if result == 'result':
            return structs.ListUserRolesResponse.from_proto(data).roles
        elif result == 'no_such_user':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NoSuchUserException(error_message)
        elif result == 'not_have_permission':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NotHavePermission(error_message)

        raise ProtocolError('Unexpected response')

    async def check_permissions(self, yandex_uid: int, permissions: list[str],
                                partner_id: int = None, agency_id: int = None) -> structs.CheckPermissionsResponse:
        if partner_id is None and agency_id is None:
            raise ProtocolError('Partner/agency id not set')

        if partner_id is not None:  # prefer partner_id
            agency_id = None

        request = request_pb2.RpcRequest(
            check_permissions=structs.CheckPermissionsRequest(
                partner_id=partner_id,
                yandex_uid=yandex_uid,
                permissions=[structs.Permission(name=permission_name) for permission_name in permissions],
                agency_id=agency_id
            ).to_proto()
        )

        result, data = await self._send_message(
            request,
            grants_pb2.CheckPermissionsOutput
        )

        if result == 'result':
            return structs.CheckPermissionsResponse.from_proto(data)
        elif result == 'no_such_user':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NoSuchUserException(error_message)
        elif result == 'no_such_permissions':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NoSuchPermissionException(error_message)

        raise ProtocolError('Unexpected response')

    async def get_accessible_agencies(self, yandex_uid: int) -> list[int]:
        request = request_pb2.RpcRequest(
            get_accessible_agencies=grants_pb2.GetAccessibleAgenciesRequest(
                yandex_uid=yandex_uid,
            )
        )

        result, data = await self._send_message(
            request,
            grants_pb2.GetAccessibleAgenciesOutput
        )

        if result == 'result':
            return structs.GetAccessibleAgenciesResponse.from_proto(data).agencies
        elif result == 'no_such_user':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NoSuchUserException(error_message)

        raise ProtocolError('Unexpected response')

    async def list_user_permissions(self, yandex_uid: int, permissions_prefix: str = None,
                                    partner_id: int = None, agency_id: int = None) -> structs.ListUserPermissionsResponse:
        if partner_id is None and agency_id is None:
            raise ProtocolError('Partner/agency id not set')

        if partner_id is not None:  # prefer partner_id
            agency_id = None

        request = request_pb2.RpcRequest(
            list_user_permissions=structs.ListUserPermissionsRequest(
                partner_id=partner_id,
                yandex_uid=yandex_uid,
                permissions_prefix=permissions_prefix,
                agency_id=agency_id
            ).to_proto()
        )

        result, data = await self._send_message(
            request,
            grants_pb2.ListUserPermissionsOutput
        )

        if result == 'result':
            return structs.ListUserPermissionsResponse.from_proto(data)
        elif result == 'no_such_user':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NoSuchUserException(error_message)
        elif result == 'no_such_permissions':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NoSuchPermissionException(error_message)

        raise ProtocolError('Unexpected response')

    async def list_available_partners(self, yandex_uid: int) -> structs.ListAvailablePartnersResponse:
        request = request_pb2.RpcRequest(
            list_available_partners=partners_pb2.ListAvailablePartnersInput(
                yandex_uid=yandex_uid,
            )
        )

        result, data = await self._send_message(
            request,
            partners_pb2.ListAvailablePartnersOutput
        )

        if result == 'result':
            return structs.ListAvailablePartnersResponse.from_proto(data)
        elif result == 'no_such_user':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NoSuchUserException(error_message)

        raise ProtocolError('Unexpected response')

    async def get_partner(self, partner_id: int, yandex_uid: int):
        request = request_pb2.RpcRequest(
            get_partner=structs.GetPartnerInput(
                yandex_uid=yandex_uid,
                partner_id=partner_id
            ).to_proto()
        )

        result, data = await self._send_message(
            request,
            partners_pb2.GetPartnerOutput
        )

        if result == 'result':
            return structs.Partner.from_proto(data)
        elif result == 'not_have_permission':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NotHavePermission(error_message)
        elif result == 'not_found':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise PartnerNotFound(error_message)

        raise ProtocolError('Unexpected response')

    async def get_partner_id(self, external_id: str, type: PartnerType, yandex_uid: int) -> structs.GetPartnerIDResponse:
        request = request_pb2.RpcRequest(
            get_partner_id=structs.GetPartnerIDInput(
                yandex_uid=yandex_uid,
                external_id=external_id,
                type=type
            ).to_proto()
        )

        result, data = await self._send_message(
            request,
            partners_pb2.GetPartnerIDOutput
        )

        if result == 'result':
            return structs.GetPartnerIDResponse.from_proto(data)
        elif result == 'not_have_permission':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NotHavePermission(error_message)
        elif result == 'not_found':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise PartnerNotFound(error_message)

        raise ProtocolError('Unexpected response')

    async def check_oauth_permissions(self, app_client_id: str, permissions: typing.List[str]) -> structs.CheckPermissionsResponse:
        request = request_pb2.RpcRequest(
            check_oauth_permissions=structs.CheckOAuthPermissionsRequest(
                app_client_id=app_client_id,
                permissions=[structs.Permission(name=permission_name) for permission_name in permissions],
            ).to_proto()
        )

        result, data = await self._send_message(
            request,
            grants_pb2.CheckOAuthPermissionsOutput
        )

        if result == 'result':
            return structs.CheckPermissionsResponse.from_proto(data)
        elif result == 'no_such_oauth_token':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NoSuchOAuthTokenException(error_message)
        elif result == 'inactive_oauth_token':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise InactiveOAuthToken(error_message)
        elif result == 'no_such_permissions':
            error_message = structs.ErrorMessageResponse.from_proto(data).message
            raise NoSuchPermissionException(error_message)

        raise ProtocolError('Unexpected response')

    async def list_partners(self, type: PartnerType = None) -> structs.ListPartnersResponse:
        request = request_pb2.RpcRequest(
            list_partners=structs.ListPartnersInput(
                type=type
            ).to_proto()
        )

        result, data = await self._send_message(
            request,
            partners_pb2.ListPartnersOutput
        )

        if result == 'result':
            return structs.ListPartnersResponse.from_proto(data)

        raise ProtocolError('Unexpected response')
