import typing

from crm.agency_cabinet.common.enum import BaseEnum
from google.protobuf.wrappers_pb2 import BoolValue
from dataclasses import dataclass, field

from crm.agency_cabinet.grants.proto import common_pb2, role_pb2, access_level_pb2, grants_pb2, user_pb2, \
    permission_pb2, partners_pb2
from smb.common.helpers import Converter
from crm.agency_cabinet.grants.common.consts import ROLE_TO_DISPLAY_NAME
from crm.agency_cabinet.grants.common.consts.partner import PartnerType
from crm.agency_cabinet.common.proto_utils import BaseStruct, safe_get_nullable_field


class AccessLevel(BaseEnum):
    ALLOW = 'ALLOW'
    DENY = 'DENY'


class AccessScopeType(BaseEnum):
    BY_ID = 'BY_ID'
    ALL = 'ALL'


access_level_converter = Converter(
    [
        (access_level_pb2.AccessLevelEnum.ALLOW, AccessLevel.ALLOW),
        (access_level_pb2.AccessLevelEnum.DENY, AccessLevel.DENY)
    ]
)

access_scope_type_converter = Converter(
    [
        (access_level_pb2.AccessScopeTypeEnum.BY_ID, AccessScopeType.BY_ID),
        (access_level_pb2.AccessScopeTypeEnum.ALL, AccessScopeType.ALL)
    ]
)


@dataclass
class CheckAccessLevelRequest(BaseStruct):
    agency_id: int
    yandex_uid: int

    @classmethod
    def from_proto(cls, message: access_level_pb2.CheckAccessLevel) -> 'CheckAccessLevelRequest':
        return cls(yandex_uid=message.yandex_uid, agency_id=message.agency_id)


@dataclass
class CheckAccessLevelResponse(BaseStruct):
    level: AccessLevel

    def to_proto(self) -> access_level_pb2.AccessLevel:
        return access_level_pb2.AccessLevel(level=access_level_converter.reversed(self.level))

    @classmethod
    def from_proto(cls, message: access_level_pb2.AccessLevel) -> 'CheckAccessLevelResponse':
        return cls(level=access_level_converter.forward(message.level))


@dataclass
class GetAccessScopeRequest(BaseStruct):
    yandex_uid: int

    @classmethod
    def from_proto(cls, message: access_level_pb2.GetAccessScope) -> 'GetAccessScopeRequest':
        return cls(yandex_uid=message.yandex_uid)


@dataclass
class GetAccessScopeResponse(BaseStruct):
    scope_type: AccessScopeType
    agencies: list[int] = field(default_factory=list)

    def to_proto(self) -> access_level_pb2.AccessScope:
        return access_level_pb2.AccessScope(
            scope_type=access_scope_type_converter.reversed(
                self.scope_type), agencies=[
                agency_id for agency_id in self.agencies])

    @classmethod
    def from_proto(cls, message: access_level_pb2.AccessScope) -> 'GetAccessScopeResponse':
        return cls(
            scope_type=access_scope_type_converter.forward(
                message.scope_type), agencies=[
                agency_id for agency_id in message.agencies])


@dataclass
class Role(BaseStruct):
    agency_id: int
    email: str
    real_ip: typing.Optional[str] = None

    @classmethod
    def from_proto(cls, message: role_pb2.Role) -> 'Role':
        return cls(
            agency_id=message.agency_id,
            email=message.email,
            real_ip=message.real_ip,
        )


@dataclass
class InternalRole(BaseStruct):
    staff_login: str
    email: str
    real_ip: typing.Optional[str] = None

    @classmethod
    def from_proto(cls, message: role_pb2.InternalRole) -> 'InternalRole':
        return cls(
            staff_login=message.staff_login,
            email=message.email,
            real_ip=message.real_ip,
        )

    def to_proto(self) -> role_pb2.InternalRole:
        return role_pb2.InternalRole(
            staff_login=self.staff_login,
            email=self.email,
            real_ip=self.real_ip,
        )


@dataclass
class GetAllInternalRolesResponse(BaseStruct):
    roles: list[InternalRole]

    @classmethod
    def from_proto(cls, message: role_pb2.InternalRoleList) -> 'GetAllInternalRolesResponse':
        return cls(roles=[InternalRole.from_proto(role) for role in message.roles])

    def to_proto(self) -> role_pb2.InternalRoleList:
        return role_pb2.InternalRoleList(roles=[role.to_proto() for role in self.roles])


@dataclass
class GetUsersPermissionsRequest(BaseStruct):
    current_yandex_uid: int
    query: str
    partner_id: typing.Optional[int] = None
    agency_id: typing.Optional[int] = None

    @classmethod
    def from_proto(cls, message: grants_pb2.GetUsersPermissions) -> 'GetUsersPermissionsRequest':
        return cls(
            partner_id=safe_get_nullable_field(message, 'partner_id'),
            current_yandex_uid=message.yandex_uid,
            query=message.query,
            agency_id=safe_get_nullable_field(message, 'agency_id')
        )

    def to_proto(self) -> grants_pb2.GetUsersPermissions:
        return grants_pb2.GetUsersPermissions(
            partner_id=self.partner_id,
            query=self.query,
            yandex_uid=self.current_yandex_uid,
            agency_id=self.agency_id
        )


@dataclass
class GetSuggestedUsersRequest(BaseStruct):
    current_yandex_uid: int
    query: str
    partner_id: typing.Optional[int] = None
    agency_id: typing.Optional[int] = None

    @classmethod
    def from_proto(cls, message: grants_pb2.GetSuggestedUsers) -> 'GetSuggestedUsersRequest':
        return cls(
            partner_id=safe_get_nullable_field(message, 'partner_id'),
            current_yandex_uid=message.yandex_uid,
            query=message.query,
            agency_id=safe_get_nullable_field(message, 'agency_id')
        )

    def to_proto(self) -> grants_pb2.GetSuggestedUsers:
        return grants_pb2.GetSuggestedUsers(
            partner_id=self.partner_id,
            query=self.query,
            yandex_uid=self.current_yandex_uid,
            agency_id=self.agency_id
        )


@dataclass
class User(BaseStruct):
    user_id: int
    name: str
    email: str
    login: str
    avatar_id: typing.Optional[str] = None
    current: bool = False

    @classmethod
    def from_proto(cls, message: user_pb2.User) -> 'User':
        return cls(
            user_id=message.user_id,
            current=message.current,
            name=message.name,
            email=message.email,
            login=message.login,
            avatar_id=message.avatar_id if message.avatar_id else None
        )

    def to_proto(self) -> user_pb2.User:
        return user_pb2.User(
            user_id=self.user_id,
            current=self.current,
            name=self.name,
            email=self.email,
            login=self.login,
            avatar_id=self.avatar_id
        )


@dataclass
class UserRole(BaseStruct):
    role_id: int
    name: str
    display_name: str = None

    def __post_init__(self):
        if not self.display_name:
            self.display_name = ROLE_TO_DISPLAY_NAME.get(self.name)

    @classmethod
    def from_proto(cls, message: role_pb2.UserRole) -> 'UserRole':
        return cls(
            role_id=message.role_id,
            name=message.name,
            display_name=message.display_name
        )

    def to_proto(self) -> role_pb2.UserRole:
        return role_pb2.UserRole(
            role_id=self.role_id,
            name=self.name,
            display_name=self.display_name
        )


@dataclass
class Permission(BaseStruct):
    name: str
    is_editable: typing.Optional[bool] = None

    @classmethod
    def from_proto(cls, message: permission_pb2.Permission) -> 'Permission':
        return cls(
            name=message.name,
            is_editable=message.is_editable.value if message.HasField('is_editable') else None
        )

    def to_proto(self) -> permission_pb2.Permission:
        return permission_pb2.Permission(
            name=self.name,
            is_editable=BoolValue(value=self.is_editable) if self.is_editable is not None else None
        )


@dataclass
class RolePermissions(BaseStruct):
    role: UserRole
    permissions: list[Permission]

    @classmethod
    def from_proto(cls, message: role_pb2.RolePermissions) -> 'RolePermissions':
        return cls(
            role=UserRole.from_proto(message.role),
            permissions=[Permission.from_proto(permission) for permission in message.permissions]
        )

    def to_proto(self) -> role_pb2.RolePermissions:
        return role_pb2.RolePermissions(
            role=self.role.to_proto(),
            permissions=[permission.to_proto() for permission in self.permissions]
        )


@dataclass
class UserPermissions(BaseStruct):
    user: User
    roles_permissions: list[RolePermissions]

    @classmethod
    def from_proto(cls, message: user_pb2.UserPermissions) -> 'UserPermissions':
        return cls(
            user=User.from_proto(message.user),
            roles_permissions=[RolePermissions.from_proto(role) for role in message.roles_permissions]
        )

    def to_proto(self) -> user_pb2.UserPermissions:
        return user_pb2.UserPermissions(
            user=self.user.to_proto(),
            roles_permissions=[role.to_proto() for role in self.roles_permissions]
        )


@dataclass
class GetUsersPermissionsResponse(BaseStruct):
    users: list[UserPermissions]

    @classmethod
    def from_proto(cls, message: grants_pb2.UsersPermissionsList) -> 'GetUsersPermissionsResponse':
        return cls(
            users=[UserPermissions.from_proto(user) for user in message.users]
        )

    def to_proto(self) -> grants_pb2.UsersPermissionsList:
        return grants_pb2.UsersPermissionsList(
            users=[user.to_proto() for user in self.users]
        )


@dataclass
class InputRole(BaseStruct):
    role_id: int
    permissions: list[Permission]

    @classmethod
    def from_proto(cls, message: grants_pb2.InputRole) -> 'InputRole':
        return cls(
            role_id=message.role_id,
            permissions=[Permission.from_proto(permission) for permission in message.permissions]
        )

    def to_proto(self) -> grants_pb2.InputRole:
        return grants_pb2.InputRole(
            role_id=self.role_id,
            permissions=[permission.to_proto() for permission in self.permissions]
        )


@dataclass
class EditUserRequest(BaseStruct):
    yandex_uid: int
    user_id: int
    roles: list[InputRole]
    partner_id: typing.Optional[int] = None
    agency_id: typing.Optional[int] = None

    @classmethod
    def from_proto(cls, message: grants_pb2.EditUserRequest) -> 'EditUserRequest':
        return cls(
            partner_id=safe_get_nullable_field(message, 'partner_id'),
            yandex_uid=message.yandex_uid,
            user_id=message.user_id,
            roles=[InputRole.from_proto(role) for role in message.roles],
            agency_id=safe_get_nullable_field(message, 'agency_id')
        )

    def to_proto(self) -> grants_pb2.EditUserRequest:
        return grants_pb2.EditUserRequest(
            partner_id=self.partner_id,
            yandex_uid=self.yandex_uid,
            user_id=self.user_id,
            roles=[role.to_proto() for role in self.roles],
            agency_id=self.agency_id
        )


@dataclass
class EditUserResponse(BaseStruct):
    is_edited: bool

    @classmethod
    def from_proto(cls, message: grants_pb2.UserEdited) -> 'EditUserResponse':
        return cls(is_edited=message.is_edited)

    def to_proto(self) -> grants_pb2.UserEdited:
        return grants_pb2.UserEdited(is_edited=self.is_edited)


@dataclass
class ErrorMessageResponse(BaseStruct):
    message: str

    @classmethod
    def from_proto(cls, _message: common_pb2.ErrorMessageResponse) -> 'ErrorMessageResponse':
        return cls(message=_message.message)

    def to_proto(self) -> common_pb2.ErrorMessageResponse:
        return common_pb2.ErrorMessageResponse(message=self.message)


@dataclass
class GetSuggestedUsersResponse(BaseStruct):
    users: list[User]

    @classmethod
    def from_proto(cls, message: grants_pb2.SuggestedUsersList) -> 'GetSuggestedUsersResponse':
        return cls(
            users=[User.from_proto(user) for user in message.users]
        )

    def to_proto(self) -> grants_pb2.SuggestedUsersList:
        return grants_pb2.SuggestedUsersList(
            users=[user.to_proto() for user in self.users]
        )


@dataclass
class ListRolesRequest(BaseStruct):
    yandex_uid: int
    partner_id: typing.Optional[int] = None
    agency_id: typing.Optional[int] = None

    @classmethod
    def from_proto(cls, message: role_pb2.ListRolesRequest) -> 'ListRolesRequest':
        return cls(
            partner_id=safe_get_nullable_field(message, 'partner_id'),
            yandex_uid=message.yandex_uid,
            agency_id=safe_get_nullable_field(message, 'agency_id')
        )

    def to_proto(self) -> role_pb2.ListRolesRequest:
        return role_pb2.ListRolesRequest(
            partner_id=self.partner_id,
            yandex_uid=self.yandex_uid,
            agency_id=self.agency_id
        )


@dataclass
class ListRolesResponse(BaseStruct):
    roles: list[RolePermissions]

    @classmethod
    def from_proto(cls, message: role_pb2.RolePermissionsList) -> 'ListRolesResponse':
        return cls(
            roles=[RolePermissions.from_proto(role) for role in message.roles]
        )

    def to_proto(self) -> role_pb2.RolePermissionsList:
        return role_pb2.RolePermissionsList(
            roles=[role.to_proto() for role in self.roles]
        )


@dataclass
class ListUserRolesRequest(BaseStruct):
    yandex_uid: int
    agency_id: typing.Optional[int] = None
    partner_id: typing.Optional[int] = None

    @classmethod
    def from_proto(cls, message: role_pb2.ListUserRolesRequest) -> 'ListUserRolesRequest':
        return cls(
            partner_id=safe_get_nullable_field(message, 'partner_id'),
            yandex_uid=message.yandex_uid,
            agency_id=safe_get_nullable_field(message, 'agency_id')
        )

    def to_proto(self) -> role_pb2.ListUserRolesRequest:
        return role_pb2.ListUserRolesRequest(
            partner_id=self.partner_id,
            yandex_uid=self.yandex_uid,
            agency_id=self.agency_id
        )


@dataclass
class ListUserPermissionsRequest(BaseStruct):
    yandex_uid: int
    permissions_prefix: str
    partner_id: typing.Optional[int] = None
    agency_id: typing.Optional[int] = None

    @classmethod
    def from_proto(cls, message: grants_pb2.ListUserPermissionsRequest) -> 'ListUserPermissionsRequest':
        return cls(
            partner_id=safe_get_nullable_field(message, 'partner_id'),
            yandex_uid=message.yandex_uid,
            permissions_prefix=message.permissions_prefix,
            agency_id=safe_get_nullable_field(message, 'agency_id')
        )

    def to_proto(self) -> grants_pb2.ListUserPermissionsRequest:
        return grants_pb2.ListUserPermissionsRequest(
            partner_id=self.partner_id,
            yandex_uid=self.yandex_uid,
            permissions_prefix=self.permissions_prefix,
            agency_id=self.agency_id
        )


@dataclass
class GetAccessibleAgenciesRequest(BaseStruct):
    yandex_uid: int

    @classmethod
    def from_proto(cls, message: grants_pb2.GetAccessibleAgenciesRequest) -> 'GetAccessibleAgenciesRequest':
        return cls(
            yandex_uid=message.yandex_uid,
        )

    def to_proto(self) -> grants_pb2.GetAccessibleAgenciesRequest:
        return grants_pb2.GetAccessibleAgenciesRequest(
            yandex_uid=self.yandex_uid,
        )


@dataclass
class GetAccessibleAgenciesResponse(BaseStruct):
    agencies: list[int]

    @classmethod
    def from_proto(cls, message: grants_pb2.AgenciesIdList) -> 'GetAccessibleAgenciesResponse':
        return cls(
            agencies=message.agencies,
        )

    def to_proto(self) -> grants_pb2.AgenciesIdList:
        return grants_pb2.AgenciesIdList(
            agencies=self.agencies,
        )


@dataclass
class ListUserRolesResponse(BaseStruct):
    roles: list[RolePermissions]

    @classmethod
    def from_proto(cls, message: role_pb2.RolePermissionsList) -> 'ListUserRolesResponse':
        return cls(
            roles=[RolePermissions.from_proto(role) for role in message.roles]
        )

    def to_proto(self) -> role_pb2.RolePermissionsList:
        return role_pb2.RolePermissionsList(
            roles=[role.to_proto() for role in self.roles]
        )


@dataclass
class ListUserPermissionsResponse(BaseStruct):
    permissions: list[Permission] = field(default_factory=list)

    @classmethod
    def from_proto(cls, message: permission_pb2.PermissionsList) -> 'ListUserPermissionsResponse':
        return cls(
            permissions=[Permission.from_proto(p) for p in message.permissions]
        )

    def to_proto(self) -> permission_pb2.PermissionsList:
        return permission_pb2.PermissionsList(
            permissions=[p.to_proto() for p in self.permissions]
        )


@dataclass
class GetPartnerInput(BaseStruct):
    yandex_uid: int
    partner_id: int

    @classmethod
    def from_proto(cls, message: partners_pb2.GetPartnerInput) -> 'GetPartnerInput':
        return cls(
            partner_id=message.partner_id,
            yandex_uid=message.yandex_uid
        )

    def to_proto(self) -> partners_pb2.GetPartnerInput:
        return partners_pb2.GetPartnerInput(
            partner_id=self.partner_id,
            yandex_uid=self.yandex_uid
        )


partner_type_converter = Converter(
    [
        (partners_pb2.PartnerType.AGENCY, PartnerType.agency),
        (partners_pb2.PartnerType.ADFOX, PartnerType.adfox),
        (partners_pb2.PartnerType.AGENCY_WO_CONTRACT, PartnerType.agency_wo_contract),
        (partners_pb2.PartnerType.DIRECT_CLIENT, PartnerType.direct_client),
        (partners_pb2.PartnerType.OTHER, PartnerType.other),
    ]
)


@dataclass
class Partner(BaseStruct):
    partner_id: int
    external_id: str
    type: PartnerType
    name: str

    @classmethod
    def from_proto(cls, message: partners_pb2.Partner) -> 'Partner':
        return cls(
            external_id=message.external_id,
            type=partner_type_converter.forward(message.type),
            partner_id=message.partner_id,
            name=message.name
        )

    def to_proto(self) -> partners_pb2.Partner:
        return partners_pb2.Partner(
            external_id=self.external_id,
            type=partner_type_converter.reversed(self.type),
            partner_id=self.partner_id,
            name=self.name
        )

    @classmethod
    def from_model(cls, model) -> typing.Optional['Partner']:
        if model is not None:
            return cls(
                external_id=model.external_id,
                type=model.type,
                partner_id=getattr(model, 'id', None) or getattr(model, 'partner_id'),
                name=model.name
            )


@dataclass
class ListAvailablePartnersInput(BaseStruct):
    yandex_uid: int

    @classmethod
    def from_proto(cls, message: partners_pb2.ListAvailablePartnersInput) -> 'ListAvailablePartnersInput':
        return cls(yandex_uid=message.yandex_uid)

    def to_proto(self) -> partners_pb2.ListAvailablePartnersInput:
        return partners_pb2.ListAvailablePartnersInput(
            yandex_uid=self.yandex_uid
        )


@dataclass
class ListAvailablePartnersResponse(BaseStruct):
    partners: list[Partner]
    size: int = None

    def __post_init__(self):
        super().__post_init__()
        if self.size is None:
            self.size = len(self.partners)

    @classmethod
    def from_proto(cls, message: partners_pb2.PartnersList) -> 'ListAvailablePartnersResponse':
        return cls(
            partners=[Partner.from_proto(partner) for partner in message.partners],
            size=message.size
        )

    def to_proto(self) -> partners_pb2.PartnersList:
        return partners_pb2.PartnersList(
            partners=[partner.to_proto() for partner in self.partners],
            size=self.size
        )


@dataclass
class ListPartnersInput(BaseStruct):
    type: PartnerType = None

    @classmethod
    def from_proto(cls, message: partners_pb2.ListPartnersInput) -> 'ListPartnersInput':
        return cls(
            type=safe_get_nullable_field(message, 'type', partner_type_converter.forward),
        )

    def to_proto(self) -> partners_pb2.ListPartnersInput:
        return partners_pb2.ListPartnersInput(
            type=partner_type_converter.reversed(self.type) if self.type else None
        )


@dataclass
class ListPartnersResponse(BaseStruct):
    partners: list[Partner]

    @classmethod
    def from_proto(cls, message: partners_pb2.PartnersList) -> 'ListPartnersResponse':
        return cls(
            partners=[Partner.from_proto(partner) for partner in message.partners]
        )

    def to_proto(self) -> partners_pb2.PartnersList:
        return partners_pb2.PartnersList(
            partners=[partner.to_proto() for partner in self.partners]
        )


@dataclass
class GetPartnerIDInput(BaseStruct):
    yandex_uid: int
    external_id: str
    type: PartnerType

    @classmethod
    def from_proto(cls, message: partners_pb2.GetPartnerIDInput) -> 'GetPartnerIDInput':
        return cls(
            yandex_uid=message.yandex_uid,
            external_id=message.external_id,
            type=partner_type_converter.forward(message.type),
        )

    def to_proto(self) -> partners_pb2.GetPartnerIDInput:
        return partners_pb2.GetPartnerIDInput(
            yandex_uid=self.yandex_uid,
            external_id=self.external_id,
            type=partner_type_converter.reversed(self.type)
        )


@dataclass
class GetPartnerIDResponse(BaseStruct):
    partner_id: int

    @classmethod
    def from_proto(cls, message: partners_pb2.PartnerID) -> 'GetPartnerIDResponse':
        return cls(
            partner_id=message.partner_id
        )

    def to_proto(self) -> partners_pb2.PartnerID:
        return partners_pb2.PartnerID(
            partner_id=self.partner_id
        )


@dataclass
class CheckOAuthPermissionsRequest(BaseStruct):
    app_client_id: str
    permissions: typing.List[Permission]

    @classmethod
    def from_proto(cls, message: grants_pb2.CheckPermissionsRequest) -> 'CheckOAuthPermissionsRequest':
        return cls(
            app_client_id=message.app_client_id,
            permissions=[Permission.from_proto(permission) for permission in message.permissions],
        )

    def to_proto(self) -> grants_pb2.CheckOAuthPermissionsRequest:
        return grants_pb2.CheckOAuthPermissionsRequest(
            app_client_id=self.app_client_id,
            permissions=[permission.to_proto() for permission in self.permissions],
        )


@dataclass
class CheckPermissionsResponse(BaseStruct):
    is_have_permissions: bool
    missed_permissions: list[Permission] = field(default_factory=list)
    partner: typing.Optional[Partner] = None

    def __post_init__(self):
        super().__post_init__()
        if self.is_have_permissions is False:
            self.partner = None

    @classmethod
    def from_proto(cls, message: grants_pb2.CheckPermissionsStatus) -> 'CheckPermissionsResponse':
        return cls(
            is_have_permissions=message.is_have_permissions,
            missed_permissions=[
                Permission.from_proto(p) for p in message.missed_permissions
            ],
            partner=safe_get_nullable_field(message, 'partner', Partner.from_proto)
        )

    def to_proto(self) -> grants_pb2.CheckPermissionsStatus:
        return grants_pb2.CheckPermissionsStatus(
            is_have_permissions=self.is_have_permissions,
            missed_permissions=[p.to_proto() for p in self.missed_permissions],
            partner=self.partner.to_proto() if self.partner else None
        )


@dataclass
class CheckPermissionsRequest(BaseStruct):
    yandex_uid: int
    permissions: list[Permission]
    partner_id: typing.Optional[int] = None
    agency_id: typing.Optional[int] = None

    @classmethod
    def from_proto(cls, message: grants_pb2.CheckPermissionsRequest) -> 'CheckPermissionsRequest':
        return cls(
            partner_id=safe_get_nullable_field(message, 'partner_id'),
            yandex_uid=message.yandex_uid,
            permissions=[Permission.from_proto(permission) for permission in message.permissions],
            agency_id=safe_get_nullable_field(message, 'agency_id')
        )

    def to_proto(self) -> grants_pb2.CheckPermissionsRequest:
        return grants_pb2.CheckPermissionsRequest(
            partner_id=self.partner_id,
            yandex_uid=self.yandex_uid,
            permissions=[permission.to_proto() for permission in self.permissions],
            agency_id=self.agency_id
        )
